<?php
/*
Plugin Name: Academic Notices Marquee
Plugin URI: https://example.com/
Description: Create a scrolling Notices / Upcoming / Announcements section for schools, colleges and universities. Adds a custom Notice post type, shortcode, Gutenberg block and Elementor widget.
Version: 1.2.0
Author: Shubham (ROI 10X)
Author URI: https://roi10x.com/
License: GPL2
Text Domain: academic-notices-marquee
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

if ( ! class_exists( 'Academic_Notices_Marquee' ) ) :

class Academic_Notices_Marquee {

    public function __construct() {
        // Register custom post type.
        add_action( 'init', array( $this, 'register_notice_post_type' ) );

        // Meta box for notice details.
        add_action( 'add_meta_boxes', array( $this, 'register_notice_meta_box' ) );
        add_action( 'save_post', array( $this, 'save_notice_meta_box' ), 10, 2 );

        // Frontend assets.
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_assets' ) );

        // Gutenberg block.
        add_action( 'init', array( $this, 'register_gutenberg_block' ) );

        // Elementor widget.
        add_action( 'elementor/widgets/register', array( $this, 'register_elementor_widget' ) );

        // Shortcode.
        add_shortcode( 'academic_notices', array( $this, 'render_notices_shortcode' ) );
    }

    /**
     * Register the "Notice" custom post type.
     */
    public function register_notice_post_type() {
        $labels = array(
            'name'                  => __( 'Notices', 'academic-notices-marquee' ),
            'singular_name'         => __( 'Notice', 'academic-notices-marquee' ),
            'menu_name'             => __( 'Notices', 'academic-notices-marquee' ),
            'name_admin_bar'        => __( 'Notice', 'academic-notices-marquee' ),
            'add_new'               => __( 'Add New', 'academic-notices-marquee' ),
            'add_new_item'          => __( 'Add New Notice', 'academic-notices-marquee' ),
            'new_item'              => __( 'New Notice', 'academic-notices-marquee' ),
            'edit_item'             => __( 'Edit Notice', 'academic-notices-marquee' ),
            'view_item'             => __( 'View Notice', 'academic-notices-marquee' ),
            'all_items'             => __( 'All Notices', 'academic-notices-marquee' ),
            'search_items'          => __( 'Search Notices', 'academic-notices-marquee' ),
            'not_found'             => __( 'No notices found.', 'academic-notices-marquee' ),
            'not_found_in_trash'    => __( 'No notices found in Trash.', 'academic-notices-marquee' ),
        );

        $args = array(
            'labels'             => $labels,
            'public'             => true,
            'publicly_queryable' => true,
            'show_ui'            => true,
            'show_in_menu'       => true,
            'query_var'          => true,
            'rewrite'            => array( 'slug' => 'notice' ),
            'capability_type'    => 'post',
            'has_archive'        => false,
            'hierarchical'       => false,
            'menu_position'      => 20,
            'menu_icon'          => 'dashicons-megaphone',
            'supports'           => array( 'title', 'editor' ),
        );

        register_post_type( 'academic_notice', $args );
    }

    /**
     * Register side meta box for Notice details (type, optional link).
     */
    public function register_notice_meta_box() {
        add_meta_box(
            'anm_notice_details',
            __( 'Notice Details', 'academic-notices-marquee' ),
            array( $this, 'render_notice_meta_box' ),
            'academic_notice',
            'side',
            'default'
        );
    }

    /**
     * Render the meta box HTML.
     */
    public function render_notice_meta_box( $post ) {
        wp_nonce_field( 'anm_save_notice_details', 'anm_notice_nonce' );

        $type      = get_post_meta( $post->ID, '_anm_notice_type', true );
        $link      = get_post_meta( $post->ID, '_anm_notice_link', true );
        $is_new    = get_post_meta( $post->ID, '_anm_is_new', true );

        if ( empty( $type ) ) {
            $type = 'notice';
        }

        ?>
        <p>
            <label for="anm_notice_type"><strong><?php esc_html_e( 'Type', 'academic-notices-marquee' ); ?></strong></label><br/>
            <select name="anm_notice_type" id="anm_notice_type" style="width:100%;">
                <option value="notice" <?php selected( $type, 'notice' ); ?>><?php esc_html_e( 'Notice', 'academic-notices-marquee' ); ?></option>
                <option value="announcement" <?php selected( $type, 'announcement' ); ?>><?php esc_html_e( 'Announcement', 'academic-notices-marquee' ); ?></option>
                <option value="upcoming" <?php selected( $type, 'upcoming' ); ?>><?php esc_html_e( 'Upcoming', 'academic-notices-marquee' ); ?></option>
            </select>
        </p>

        <p>
            <label for="anm_notice_link"><strong><?php esc_html_e( 'Optional link (PDF / page URL)', 'academic-notices-marquee' ); ?></strong></label>
            <input type="url" name="anm_notice_link" id="anm_notice_link" value="<?php echo esc_attr( $link ); ?>" style="width:100%;" placeholder="https://example.com/file.pdf" />
        </p>

        <p>
            <label for="anm_is_new">
                <input type="checkbox" name="anm_is_new" id="anm_is_new" value="1" <?php checked( $is_new, '1' ); ?> />
                <?php esc_html_e( 'Show "New" badge', 'academic-notices-marquee' ); ?>
            </label>
        </p>
        <?php
    }

    /**
     * Save meta box values.
     */
    public function save_notice_meta_box( $post_id, $post ) {
        // Check nonce.
        if ( ! isset( $_POST['anm_notice_nonce'] ) || ! wp_verify_nonce( $_POST['anm_notice_nonce'], 'anm_save_notice_details' ) ) {
            return;
        }

        // Autosave?
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }

        // Check permissions.
        if ( 'academic_notice' !== $post->post_type || ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }

        $type   = isset( $_POST['anm_notice_type'] ) ? sanitize_text_field( $_POST['anm_notice_type'] ) : 'notice';
        $link   = isset( $_POST['anm_notice_link'] ) ? esc_url_raw( $_POST['anm_notice_link'] ) : '';
        $is_new = isset( $_POST['anm_is_new'] ) ? '1' : '0';

        update_post_meta( $post_id, '_anm_notice_type', $type );
        update_post_meta( $post_id, '_anm_notice_link', $link );
        update_post_meta( $post_id, '_anm_is_new', $is_new );
    }

    /**
     * Enqueue frontend CSS/JS.
     */
    public function enqueue_assets() {
        wp_enqueue_style(
            'academic-notices-marquee-style',
            plugins_url( 'assets/css/anm-style.css', __FILE__ ),
            array(),
            '1.2.0'
        );
        wp_enqueue_script(
            'academic-notices-marquee-script',
            plugins_url( 'assets/js/anm-script.js', __FILE__ ),
            array( 'jquery' ),
            '1.2.0',
            true
        );
    }

    /**
     * Register Gutenberg block.
     */
    public function register_gutenberg_block() {
        if ( ! function_exists( 'register_block_type' ) ) {
            return;
        }

        wp_register_script(
            'academic-notices-marquee-block-editor',
            plugins_url( 'assets/js/anm-block.js', __FILE__ ),
            array( 'wp-blocks', 'wp-element', 'wp-components', 'wp-editor', 'wp-i18n', 'wp-server-side-render' ),
            '1.2.0',
            true
        );

        register_block_type(
            'academic-notices/marquee',
            array(
                'editor_script'   => 'academic-notices-marquee-block-editor',
                'render_callback' => array( $this, 'render_notices_block' ),
                'attributes'      => array(
                    'heading'      => array(
                        'type'    => 'string',
                        'default' => __( 'Notices', 'academic-notices-marquee' ),
                    ),
                    'type'         => array(
                        'type'    => 'string',
                        'default' => 'all',
                    ),
                    'limit'        => array(
                        'type'    => 'number',
                        'default' => 10,
                    ),
                    'wrapperClass' => array(
                        'type'    => 'string',
                        'default' => '',
                    ),
                    'showHeading'  => array(
                        'type'    => 'boolean',
                        'default' => true,
                    ),
                ),
            )
        );
    }

    /**
     * Render callback for Gutenberg block.
     */
    public function render_notices_block( $attributes, $content ) {
        $atts = array(
            'heading'       => isset( $attributes['heading'] ) ? $attributes['heading'] : __( 'Notices', 'academic-notices-marquee' ),
            'type'          => isset( $attributes['type'] ) ? $attributes['type'] : 'all',
            'limit'         => isset( $attributes['limit'] ) ? $attributes['limit'] : 10,
            'wrapper_class' => isset( $attributes['wrapperClass'] ) ? $attributes['wrapperClass'] : '',
            'show_heading'  => ( isset( $attributes['showHeading'] ) && false === $attributes['showHeading'] ) ? 'no' : 'yes',
        );

        return $this->render_notices_shortcode( $atts );
    }

    /**
     * Register Elementor widget.
     */
    public function register_elementor_widget( $widgets_manager ) {
        if ( ! class_exists( '\Elementor\Widget_Base' ) ) {
            return;
        }

        require_once plugin_dir_path( __FILE__ ) . 'includes/class-anm-elementor-widget.php';

        if ( class_exists( '\ANM_Elementor_Notices_Widget' ) ) {
            $widgets_manager->register( new \ANM_Elementor_Notices_Widget() );
        }
    }

    /**
     * Render notices via shortcode.
     *
     * Usage examples:
     * [academic_notices]
     * [academic_notices heading="Announcements"]
     * [academic_notices heading="Upcoming Events" wrapper_class="my-custom-class" show_heading="no"]
     */
    public function render_notices_shortcode( $atts ) {
        $atts = shortcode_atts(
            array(
                'heading'       => __( 'Notices', 'academic-notices-marquee' ),
                'type'          => 'all', // notice, announcement, upcoming, all.
                'limit'         => 10,
                'wrapper_class' => '',
                'show_heading'  => 'yes', // yes or no.
            ),
            $atts,
            'academic_notices'
        );

        $wrapper_classes = array( 'anm-box' );
        if ( ! empty( $atts['wrapper_class'] ) ) {
            $wrapper_classes[] = sanitize_html_class( $atts['wrapper_class'] );
        }

        $query_args = array(
            'post_type'      => 'academic_notice',
            'posts_per_page' => intval( $atts['limit'] ),
            'post_status'    => 'publish',
            'orderby'        => 'date',
            'order'          => 'DESC',
        );

        if ( 'all' !== $atts['type'] ) {
            $query_args['meta_query'] = array(
                array(
                    'key'   => '_anm_notice_type',
                    'value' => sanitize_text_field( $atts['type'] ),
                ),
            );
        }

        $loop = new WP_Query( $query_args );

        ob_start();

        if ( $loop->have_posts() ) : ?>
            <div class="<?php echo esc_attr( implode( ' ', $wrapper_classes ) ); ?>">
                <?php if ( 'no' !== strtolower( $atts['show_heading'] ) ) : ?>
                    <div class="anm-header">
                        <span class="anm-header-title"><?php echo esc_html( $atts['heading'] ); ?></span>
                    </div>
                <?php endif; ?>

                <div class="anm-marquee-wrapper">
                    <marquee behavior="scroll" direction="up" scrollamount="2" onmouseover="this.stop();" onmouseout="this.start();">
                        <ul class="anm-list">
                            <?php
                            while ( $loop->have_posts() ) :
                                $loop->the_post();
                                $type   = get_post_meta( get_the_ID(), '_anm_notice_type', true );
                                $link   = get_post_meta( get_the_ID(), '_anm_notice_link', true );
                                $is_new = get_post_meta( get_the_ID(), '_anm_is_new', true );
                                ?>
                                <li class="anm-item anm-type-<?php echo esc_attr( $type ? $type : 'notice' ); ?>">
                                    <?php if ( $link ) : ?>
                                        <a href="<?php echo esc_url( $link ); ?>" target="_blank" rel="noopener">
                                            <?php the_title(); ?>
                                        </a>
                                    <?php else : ?>
                                        <span class="anm-item-title"><?php the_title(); ?></span>
                                    <?php endif; ?>

                                    <?php if ( '1' === $is_new ) : ?>
                                        <span class="anm-badge"><?php esc_html_e( 'New', 'academic-notices-marquee' ); ?></span>
                                    <?php endif; ?>
                                </li>
                            <?php endwhile; ?>
                        </ul>
                    </marquee>
                </div>
            </div>
        <?php
        endif;

        wp_reset_postdata();

        return ob_get_clean();
    }
}

endif;

// Initialize plugin.
new Academic_Notices_Marquee();
