<?php

namespace App\Http\Controllers;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use App\Models\User;
use App\Models\Admins;
use DataTables;
use Validator;
use Session;
use Auth;
use Hash;

class AdminController extends Controller{

    public function __construct(){
        $this->middleware('permission:view-admin', ['only' => ['index','show']]);
        $this->middleware('permission:create-admin', ['only' => ['create','store']]);
        $this->middleware('permission:update-admin', ['only' => ['edit','store']]);
        $this->middleware('permission:delete-admin', ['only' => ['destroy']]);
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request){
        try {
            $data = [];
            $data['page_title']        = 'Admins List';
            if($request->ajax()){
                $admins = Admins::query()->with('user');
                return DataTables::eloquent($admins)
                    ->addColumn('action', function($admins) {
                        $action      = '';
                        if(Auth::user()->can('view-admin')){
                            $action .= '<a href="'.route("admins.view",$admins->id).'" title="View Admin"><i class="menu-icon tf-icons ti ti-eye"></i></a>&nbsp;';
                        }
                        if(Auth::user()->can('update-admin')){
                            $action .= '<a href="'.route("admins.edit",$admins->id).'" title="Edit Admin"><i class="menu-icon tf-icons ti ti-edit"></i></a>&nbsp;';
                        }
                        if(Auth::user()->can('delete-admin')){
                            $action .= '<a href="javascript:void(0);" id="tutor_id_'.$admins->id.'" title="Delete Admin" data-id="'.$admins->id.'" onclick="deleteRecord(this,'.$admins->id.');"><i class="menu-icon tf-icons ti ti-trash"></i></a>&nbsp;';
                        }
                    
                        return $action;
                    })
                    ->addColumn('email', function($admins) {
                        return $admins->user->email;
                    })
                    ->filterColumn('email', function ($query, $keyword) {
                        $query->whereHas('user', function ($sub_q) use ($keyword) {
                            $sub_q->where('email', 'like', "%$keyword%");
                        });
                    })
                    ->addColumn('name', function($admins) {
                        return $admins->first_name.' '.$admins->last_name;
                    })
                    ->filterColumn('name', function ($query, $keyword) {
                        $query->where('first_name', 'LIKE', "{$keyword}%")
                            ->where('last_name', 'LIKE', "{$keyword}%");
                    })
                    ->editColumn('created_at', function($users) {
                        return ($users->created_at!=null)?date(config('constant.date_format'), strtotime($users->created_at)):'-';
                    })
                    ->editColumn('updated_at', function($users) {
                        return ($users->updated_at!=null)?date(config('constant.date_format'), strtotime($users->updated_at)):'-';
                    })
                    ->filterColumn('created_at', function ($query, $keyword) {
                        $query->whereRaw("DATE_FORMAT(created_at,'%d-%m-%Y %h:%i %r') like ?", ["%$keyword%"]);
                    })
                    ->filterColumn('updated_at', function ($query, $keyword) {
                        $query->whereRaw("DATE_FORMAT(updated_at,'%d-%m-%Y %h:%i %r') like ?", ["%$keyword%"]);
                    })
                    ->rawColumns(['action'])
                    ->make(true);
            }
            return view('admins.index',$data);
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    public function exists(Request $request){
        if(isset($request->id) && $request->id!=''){
            $result = User::where('id', '!=' , $request->id)->where('email',$request->email)->count();
        }else{
            $result = User::where('email',$request->email)->count();
        }
        if($result>0){
            return response()->json(false);
        }else{
            return response()->json(true);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(){
        try {
            $data['page_title']         = 'Add Admin';
            $data['roles']              = Role::whereNull('deleted_at')->get();
            return view('admins.add',$data);
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request){
        try {
            $rules          = [
                'first_name'              => 'required|string|max:255',
                'last_name'              => 'required|string|max:255',
            ];

            if (!isset($request->id)) {
                $rules['email']         = 'required|string|email|max:255|unique:users';
                $rules['password']         = 'required|string|min:8';
            }

            $messages       = [
                'first_name.required'         => 'The first name field is required.',
                'last_name.required'         => 'The last name field is required.',
                'email.required'        => 'The email field is required.',
                'password.required' 	=> 'The password field is required.',
            ];
            $validator      = Validator::make($request->all(), $rules,$messages);
            if ($validator->fails()) {
                if (isset($request->id) && !empty($request->id)) {
                    return redirect()->route('admins.edit',$request->id)
                                ->withErrors($validator)
                                ->withInput();
                }else{
                    return redirect()->route('admins.add')
                                ->withErrors($validator)
                                ->withInput();
                }
            }else{
                $name                       = $request->first_name." ".$request->last_name;

                $role = Role::where('name', 'admin')->whereNull('deleted_at')->first();

                if (isset($request->id) && !empty($request->id)) {
                    $id                     = $request->id;
                    $admin                  = Admins::where('id',$id)->first();
                    $user                   = $admin->user;
                    $user->name             = trim($name);
                    if($request->email != $user->email){
                        $user->email            = trim($request->email);
                    }
                    $user->updated_by       = Auth::user()->id;
                    $user->updated_at       = date("Y-m-d H:i:s");
                    if($user->save()){
                        $user->syncRoles($role);
                        $admin->first_name                   = $request->first_name;
                        $admin->last_name                    = $request->last_name;
                        $admin->user_id                      = $user->id;
                        $admin->updated_by                   = Auth::user()->id;
                        $admin->updated_at                   = date("Y-m-d H:i:s");
                        $admin->save();
                        
                        Session::flash('alert-message','Admin updated successfully.');
                        Session::flash('alert-class','success');
                        return redirect()->route('admins');
                    }else{
                        Session::flash('alert-message','Admin updated unsuccessfully.');
                        Session::flash('alert-class','error');
                        return redirect()->route('admins.edit',$id);
                    }
                }else{
                    $user                       = new User();
                    $user->name                 = trim($name);
                    $user->password             = bcrypt($request->password);
                    $user->email                = trim($request->email);
                    $user->email_verified_at    = date("Y-m-d H:i:s");
                    $user->created_by           = Auth::user()->id;
                    $user->created_at           = date("Y-m-d H:i:s");
                    if($user->save()){
                        $user->assignRole($role);

                        $admin                               = new Admins;
                        $admin->first_name                   = $request->first_name;
                        $admin->last_name                    = $request->last_name;
                        $admin->user_id                      = $user->id;
                        $admin->created_by                   = Auth::user()->id;
                        $admin->created_at                   = date("Y-m-d H:i:s");
                        $admin->save();

                        Session::flash('alert-message','Admin added successfully.');
                        Session::flash('alert-class','success');
                        return redirect()->route('admins');
                    }else{
                        Session::flash('alert-message','Admin added unsuccessfully.');
                        Session::flash('alert-class','error');
                        return redirect()->route('admins.add');
                    }
                }
            }
        }catch (\Exception $e) {
            Session::flash('alert-message',$e->getMessage());
            Session::flash('alert-class','error');
            return redirect()->route('admins');
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id){
        try {
            $data['page_title']         = 'View Admin';
            $admin                      = Admins::where('id',$id)->first();
            if($admin){
                //$data['user']   = $user;
                $data['admin']   = $admin;
                return view('admins.view',$data);
            }else{
                return abort(404);
            }
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id){
        try {
            $data['page_title']         = 'Edit Admin';
            $admin                       = Admins::where('id',$id)->first();
            if($admin){
                $data['roles']           = Role::whereNull('deleted_at')->get();
                $data['admin']           = $admin;
                return view('admins.edit',$data);
            }else{
                return abort(404);
            }
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request){
        if ($request->ajax()) {
            try {
                $admin                                   = Admins::where('id',$request->id)->first();
                if(!is_null($admin)){
                    if(Auth::user()->id != $admin->user->id){
                        $admin->disableLogging();
                        $admin->deleted_by           = Auth::user()->id;
                        $admin->save();
                        $admin->enableLogging();
                        if($admin->delete() && $admin->user->delete()){
                            $response['success']    = true;
                            $response['message']    = "Admin deleted successfully.";
                        }else{
                            $response['success']    = false;
                            $response['message']    = "Admin deleted unsuccessfully.";
                        }
                    }else{
                        $response['success']        = false;
                        $response['message']        = "Can't delete Admin because ".$admin->user()->name." is active.";
                    }
                }else{
                    $response['success']                = false;
                    $response['message']                = "Admin record not found.";
                }
            } catch (\Exception $e) {
                $response['success']                    = false;
                $response['message']                    = $e->getMessage();
            }
            return response()->json($response);
        }else{
            return abort(404);
        }
    }

  
}
