<?php

namespace App\Http\Controllers;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use App\Models\User;
use App\Models\Batches;
use App\Models\Meeting;
use App\Models\Students;
use App\Models\Tutors;
use App\Models\Lessons;
use App\Models\BatchStudents;
use App\Mail\BatchCreatedTutorMail;
use App\Mail\BatchCreatedParentMail;
use App\Mail\FeeReminderMail;
use Carbon\Carbon;
use DataTables;
use Validator;
use Session;
use Auth;
use Hash;
use Mail;

class BatchController extends Controller{

    public function __construct(){
        $this->middleware('permission:view-batch', ['only' => ['index','show']]);
        $this->middleware('permission:create-batch', ['only' => ['create','store']]);
        $this->middleware('permission:update-batch', ['only' => ['edit','store']]);
        $this->middleware('permission:delete-batch', ['only' => ['destroy']]);
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request){
        try {
            $data = [];
            $data['page_title']        = 'Batch List';
            if($request->ajax()){
                $user = Auth::user();
                $batches = null;
                if(Auth::user()->role_name == RoleTutor){
                    $tutor = Tutors::where('user_id', $user->id)->first();
                    $batches = Batches::where('tutor_id', $tutor->id)->whereNull('deleted_at');
                }
                else if(Auth::user()->role_name == RoleStudent){
                    $student = Students::where('user_id', $user->id)->first();
                    $batchIds = BatchStudents::where('students_id', $student->id)->get()->pluck('batches_id')->toArray();
                    $batches = Batches::whereIn('id', $batchIds)->whereNull('deleted_at');
                }
                else{
                    $batches = Batches::whereNull('deleted_at');
                }
                
                return DataTables::eloquent($batches)
                    ->addColumn('action', function($batches) {
                        $action      = '';
                        if(Auth::user()->can('create-batch')){
                            $action .= '<a href="'.route("batches.student",$batches->id).'" title="Add Student to Batch"><i class="menu-icon tf-icons ti ti-user-plus"></i></a>&nbsp;';
                        }
                        if(Auth::user()->can('create-batch')){
                            $action .= '<a href="'.route("batches.lesson",$batches->id).'" title="Add Lesson to Batch"><i class="menu-icon tf-icons ti ti-calendar-plus"></i></a>&nbsp;';
                        }
                        if(Auth::user()->can('view-batch')){
                            $action .= '<a href="'.route("batches.view",$batches->id).'" title="View Batch"><i class="menu-icon tf-icons ti ti-eye"></i></a>&nbsp;';
                        }
                        if(Auth::user()->can('update-batch')){
                            $action .= '<a href="'.route("batches.edit",$batches->id).'" title="Edit Batch"><i class="menu-icon tf-icons ti ti-edit"></i></a>&nbsp;';
                        }
                        if(Auth::user()->can('delete-batch')){
                            $action .= '<a href="javascript:void(0);" id="tutor_id_'.$batches->id.'" data-id="'.$batches->id.'" title="Delete Batch" onclick="deleteRecord(this,'.$batches->id.');"><i class="menu-icon tf-icons ti ti-trash"></i></a>&nbsp;';
                        }
                    
                        return $action;
                    })
                    ->addColumn('tutor_id', function($batches) {
                        return $batches->tutor->first_name." ".$batches->tutor->last_name;
                    })
                    ->filterColumn('tutor_id', function ($query, $keyword) {
                        $query->whereHas('tutor', function ($q) use ($keyword) {
                            $q->where('first_name', 'LIKE', "{$keyword}%")
                            ->where('last_name', 'LIKE', "{$keyword}%");
                        });
                    })
                    ->addColumn('lessons', function($batches) {
                        return $batches->lessons->count(). " Lessons";
                    })
                    ->addColumn('students', function($batches) {
                        return $batches->students->count()." Students";
                    })
                    ->editColumn('created_at', function($users) {
                        return ($users->created_at!=null)?date(config('constant.date_format'), strtotime($users->created_at)):'-';
                    })
                    ->editColumn('updated_at', function($users) {
                        return ($users->updated_at!=null)?date(config('constant.date_format'), strtotime($users->updated_at)):'-';
                    })
                    ->filterColumn('created_at', function ($query, $keyword) {
                        $query->whereRaw("DATE_FORMAT(created_at,'%d-%m-%Y %h:%i %r') like ?", ["%$keyword%"]);
                    })
                    ->filterColumn('updated_at', function ($query, $keyword) {
                        $query->whereRaw("DATE_FORMAT(updated_at,'%d-%m-%Y %h:%i %r') like ?", ["%$keyword%"]);
                    })
                    ->rawColumns([])
                    ->make(true);
            }
            return view('batches.index',$data);
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    public function exists(Request $request){
        if(isset($request->id) && $request->id!=''){
            $result = Batches::where('id', '!=' , $request->id)->count();
        }else{
            $result = Batches::where('id',$request->id)->count();
        }
        if($result>0){
            return response()->json(false);
        }else{
            return response()->json(true);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(){
        try {
            $data['page_title']         = 'Add Batch';
            $data['tutors']             = Tutors::whereNull('deleted_at')->get();
            $data['students']           = Students::whereNull('deleted_at')->get();
            $data['days'] = [ 
                ['id'=>'0', 'name'=>'SUN'],
                ['id'=>'1', 'name'=>'MON'],
                ['id'=>'2', 'name'=>'TUE'],
                ['id'=>'3', 'name'=>'WED'],
                ['id'=>'4', 'name'=>'THU'],
                ['id'=>'5', 'name'=>'FRI'],
                ['id'=>'6', 'name'=>'SAT']
            ];
            $data['timings'] = [ 30, 35, 40, 45, 50, 55, 60 ];

            return view('batches.add',$data);
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request){
        try {
            $rules          = [
                'name'                      => 'required|string|max:255',
                'description'               => 'required|string|max:255',
                "tutor_id"      			=> "required",
                //"students"      			=> "required|array|min:1"
            ];

            $messages       = [
                'name.required'                 => 'The name field is required.',
                'description.required'          => 'The description field is required.',
                'tutor_id.required'             => 'The tutor name field is required.',
                //'students.required'             => 'The students field is required.',
            ];

            if(isset($request->days) && isset($request->id)){
                $rules['days'] = "required";
                $messages['days.required'] = "Please select lesson days";
            }

            if (!isset($request->id) || empty($request->id)) {
                $rules = array_merge($rules, [
                    "start_date"      			=> "required",
                    "charge"      		        => "required",
                    "duration"                  => "required",
                    "days"      			    => "required|array|min:1"
                ]); 

                $messages = array_merge($rules, [
                    'start_date.required'           => 'The start date field is required.',
                    'charge.required' 	            => 'The per lesson charge field is required.',
                    'duration.required'             => 'The lesson duration field is required.',
                    'days.required'                 => 'The days field is required.'
                ]); 
            }

            $validator      = Validator::make($request->all(), $rules,$messages);

            if ($validator->fails()) {
                if (isset($request->id) && !empty($request->id)) {
                    return redirect()->route('batches.edit',$request->id)
                                ->withErrors($validator)
                                ->withInput();
                }else{
                    return redirect()->route('batches.add')
                                ->withErrors($validator)
                                ->withInput();
                }
            }else{
                if (isset($request->id) && !empty($request->id)) {
                    $id                          = $request->id;
                    $batch                       = Batches::where('id',$id)->first();
                    $batchStudents               = $batch->students()->get()->pluck('id')->toArray();
                    $students                    = $request->students;

                    $batch->name                 = $request->name;
                    $batch->description          = $request->description;
                    $batch->tutor_id             = $request->tutor_id;
                    $batch->updated_by           = Auth::user()->id;
                    $batch->updated_at           = date("Y-m-d H:i:s");
                    $batch->batch_type             = $request->batch_type;
                    if($batch->save()){
                        $batch->students()->detach($batchStudents);
                        $batch->students()->attach($students);
                        if($request->reset_all_lessons) {
                            activity()->disableLogging();
                            $batch->lessons()->delete();
                            
                            $start_date         = Carbon::createFromTimestamp(strtotime($request->start_date));
                            $lesson_numbers     = $request->lesson_numbers;
                            $max_lessons        = $request->max_lessons;
                            $charge             = $request->charge;
                            $duration           = $request->duration;
                            $days               = $request->days;
                            $is_report_required = $request->is_report_required;
                            $lesson_dates       = [];


                            $date = $start_date->copy();
                            while(count($lesson_dates) < $lesson_numbers){
                                if($days){
                                    if(in_array(Carbon::createFromTimestamp(strtotime($date))->dayOfWeek, $days)){
                                        $e = Carbon::createFromFormat('Y-m-d H:i:s', $date);
                                            $t = $e->format('Y-m-d')." ".$start_date->toTimeString();
                                            array_push($lesson_dates, [ 
                                                'start_date'=> $date->format('Y-m-d H:i:s'),
                                                'end_date'=> Carbon::createFromFormat('Y-m-d H:i:s', $t)
                                                    ->addMinutes($duration)
                                                    ->format('Y-m-d H:i:s')
                                                ]);
                                    }
                                }
                                
                                $date = $date->addDay();
                            }

                            $i = 1;
                            foreach($lesson_dates as $date){        
                                $lesson                       = new Lessons();
                                $lesson->name                 = "Lesson ".$i;
                                $lesson->start_date           = $date['start_date'];
                                $lesson->end_date             = $date['end_date'];
                                $lesson->charge               = $request->charge;
                                $lesson->status               = LessonStatusPlanned;
                                $lesson->batch_id             = $batch->id;
                                $lesson->is_report_required   = $is_report_required;
                                $lesson->created_by           = Auth::user()->id;
                                $lesson->created_at           = date("Y-m-d H:i:s");
                                $lesson->save();

                                /*
                                $meeting                       = new Meeting();
                                $meeting->meeting_id           = $this->random_string(10);
                                $meeting->lesson_id            = $lesson->id;
                                $meeting->user_id              = $tutor->user->id;
                                $meeting->status               = 'active';
                                $meeting->date                 = date('Y-m-d', strtotime($date['start_date']));
                                $meeting->time                 = date('H:i', strtotime($date['start_date']));
                                $meeting->created_at           = date("Y-m-d H:i:s");
                                $meeting->save();
                                */
                                $i++;
                            }
                            
                            activity()->enableLogging();
                            

                        }
                        else{
                            $lesson_ids = $_POST['lesson']['id'];
                            $lesson_charges = $_POST['lesson']['charge'];
                            
                            for($i=0; $i<=count($lesson_ids)-1;$i++){
                                $lesson = Lessons::where('id', $lesson_ids[$i])->first();
                                if($lesson){
                                    $lesson->charge = $lesson_charges[$i];
                                    $lesson->save();
                                }
                            }
                        }
                        Session::flash('alert-message','Batch updated successfully.');
                        Session::flash('alert-class','success');
                        return redirect()->route('batches');
                    }else{
                        Session::flash('alert-message','Batch updated unsuccessfully.');
                        Session::flash('alert-class','error');
                        return redirect()->route('batches.edit',$id);
                    }
                }else{
                    $start_date         = Carbon::createFromTimestamp(strtotime($request->start_date));
                    $lesson_numbers     = $request->lesson_numbers;
                    $max_lessons        = $request->max_lessons;
                    $charge             = $request->charge;
                    $duration           = $request->duration;
                    $students           = $request->students;
                    $days               = $request->days;
                    $is_report_required = $request->is_report_required;
                    $lesson_dates       = [];


                    $date = $start_date->copy();
                    while(count($lesson_dates) < $lesson_numbers){
                        if(in_array(Carbon::createFromTimestamp(strtotime($date))->dayOfWeek, $days)){
                            $e = Carbon::createFromFormat('Y-m-d H:i:s', $date);
                                $t = $e->format('Y-m-d')." ".$start_date->toTimeString();
                                array_push($lesson_dates, [ 
                                    'start_date'=> $date->format('Y-m-d H:i:s'),
                                    'end_date'=> Carbon::createFromFormat('Y-m-d H:i:s', $t)
                                        ->addMinutes($duration)
                                        ->format('Y-m-d H:i:s')
                                    ]);
                        }
                        
                        $date = $date->addDay();
                    }

                    $tutor                       = Tutors::where('id', $request->tutor_id)->first();
                    $batch                       = new Batches();
                    $batch->name                 = $request->name;
                    $batch->description          = $request->description;
                    $batch->tutor_id             = $request->tutor_id;
                    $batch->created_by           = Auth::user()->id;
                    $batch->created_at           = date("Y-m-d H:i:s");
                    $batch->batch_type             = $request->batch_type;
                    if($batch->save()){
                        $batch->students()->attach($students);

                        activity()->disableLogging();

                        $i = 1;
                        foreach($lesson_dates as $date){        
                            $lesson                       = new Lessons();
                            $lesson->name                 = "Lesson ".$i;
                            $lesson->start_date           = $date['start_date'];
                            $lesson->end_date             = $date['end_date'];
                            $lesson->charge               = $request->charge;
                            $lesson->status               = LessonStatusPlanned;
                            $lesson->batch_id             = $batch->id;
                            $lesson->is_report_required   = $is_report_required;
                            $lesson->created_by           = Auth::user()->id;
                            $lesson->created_at           = date("Y-m-d H:i:s");
                            $lesson->save();
                            /*
                            $meeting                       = new Meeting();
                            $meeting->meeting_id           = $this->random_string(10);
                            $meeting->lesson_id            = $lesson->id;
                            $meeting->user_id              = $tutor->user->id;
                            $meeting->status               = 'active';
                            $meeting->date                 = date('Y-m-d', strtotime($date['start_date']));
                            $meeting->time                 = date('H:i', strtotime($date['start_date']));
                            $meeting->created_at           = date("Y-m-d H:i:s");
                            $meeting->save();
                            */
                            $i++;
                        }

                        activity()->enableLogging();
                        
                        if(getSetting('ALLOW_MAIL_FEATURE') == 1){
                            //Mail to Tutor
                            Mail::to(trim($batch->tutor->user->email))->send(new BatchCreatedTutorMail($batch));
                            
                            if($students){
                                foreach($students as $id){
                                    $student = Students::where('id', $id)->first();
    
                                    //Mail to Parent
                                    Mail::to(trim($student->parent_email))->send(new BatchCreatedParentMail($batch, $student));
                                }
                            }
                        }

                        Session::flash('alert-message','Batch added successfully.');
                        Session::flash('alert-class','success');
                        return redirect()->route('batches');
                    }else{
                        Session::flash('alert-message','Batch added unsuccessfully.');
                        Session::flash('alert-class','error');
                        return redirect()->route('batches.add');
                    }
                }
            }
        }catch (\Exception $e) {
            Session::flash('alert-message',$e->getMessage());
            Session::flash('alert-class','error');
            return redirect()->route('batches');
        }
    }

    private function random_string($length) {
        $key = '';
        $keys = array_merge(range(0, 9), range('a', 'z'));
    
        for ($i = 0; $i < $length; $i++) {
            $key .= $keys[array_rand($keys)];
        }
    
        return $key;
    }

    private function generateDateRange(Carbon $start_date, $number)
    {
        $dates = [];

        for($date = $start_date->copy(), $i=1; $i<=$number; $date->addDay()) {
            $dates[] = $date->format('Y-m-d H:i:s');
            $i++;
        }

        return $dates;
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id){
        try {
            $data['page_title']         = 'View Batch';
            $batch                      = Batches::where('id',$id)->first();
            if($batch){
                //$data['user']   = $user;
                $data['batch']   = $batch;
                return view('batches.view',$data);
            }else{
                return abort(404);
            }
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id){
        try {
            $data['page_title']         = 'Edit Batch';
            $data['days'] = [ 
                ['id'=>'0', 'name'=>'SUN'],
                ['id'=>'1', 'name'=>'MON'],
                ['id'=>'2', 'name'=>'TUE'],
                ['id'=>'3', 'name'=>'WED'],
                ['id'=>'4', 'name'=>'THU'],
                ['id'=>'5', 'name'=>'FRI'],
                ['id'=>'6', 'name'=>'SAT']
            ];
            $data['timings'] = [ 30, 35, 40, 45, 50, 55, 60 ];
            $batch                      = Batches::where('id',$id)->first();
            $total_planned_lessons      = Lessons::where(['batch_id'=>$batch->id, 'status'=> LessonStatusPlanned])->count();

            if($batch){
                $data['batch']           = $batch;
                $data['tutors']             = Tutors::whereNull('deleted_at')->get();
                $data['students']           = Students::whereNull('deleted_at')->get();
                $data['batchStudents']      = $batch->students()->get()->pluck('id')->toArray();
                $data['show_date_edit_section'] = $total_planned_lessons == $batch->lessons()->count();
                $data['total_lessons']          = $batch->lessons()->count();
                $data['start_date']          = $batch->lessons()->first()->start_date;
                $data['charge']          = $batch->lessons()->first()->charge;

                return view('batches.edit',$data);
            }else{
                return abort(404);
            }
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function lesson($id){
        try {
            $batch                       = Batches::where('id',$id)->first();
            $data['page_title']          = 'Add Lesson';
            $data['header']              = 'Add Lesson to Batch '.$batch->name;
            if($batch){
                $data['batch']           = $batch;
                return view('batches.add_lesson',$data);
            }else{
                return abort(404);
            }
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function storeLesson(Request $request){
        try {
            $batch_id = $request->id;
            $rules          = [
                'name'                      => 'required|string|max:255',
                "start_date"      			=> "required",
                "end_date"      			=> "required",
            ];

            $messages       = [
                'name.required'                 => 'The name field is required.',
                'start_date.required'           => 'The start date field is required.',
                'end_date.required'             => 'The end_date field is required.',
            ];
            $validator      = Validator::make($request->all(), $rules,$messages);
            if ($validator->fails()) {
                return redirect()->route('batches.lesson',$batch_id)
                                ->withErrors($validator)
                                ->withInput();
            }else{
                $start_date         = Carbon::createFromTimestamp(strtotime($request->start_date));
                $end_date           = Carbon::createFromTimestamp(strtotime($request->end_date));

                $existing_lesson = Lessons::where('batch_id', $batch_id)->where('name', '=', $request->name)->first();
                if ($existing_lesson) {
                    Session::flash('alert-message','A Lesson already exists with provided name.');
                    Session::flash('alert-class','error');
                    return redirect()->route('batches.lesson', $batch_id);
                }

                $existing_lesson = Lessons::where('batch_id', $batch_id)->whereBetween('start_date', [ $start_date->format('Y-m-d H:i:s'), 
                    $end_date->format('Y-m-d H:i:s')])->first();

                if ($existing_lesson) {
                    Session::flash('alert-message','A Lesson already exists with provided time slot.');
                    Session::flash('alert-class','error');
                    return redirect()->route('batches.lesson', $batch_id);
                }

                $batch                        = Batches::where('id', $batch_id)->first();
                $default_lesson               = $batch->lessons->get(0);

                $lesson                       = new Lessons();
                $lesson->name                 = $request->name;
                $lesson->start_date           = $start_date->format("Y-m-d H:i:s");
                $lesson->end_date             = $end_date->format("Y-m-d H:i:s");
                $lesson->status               = LessonStatusPlanned;
                $lesson->is_report_required   = $request->is_report_required;
                $lesson->charge               = $request->charge;
                $lesson->batch_id             = $batch_id;
                $lesson->created_by           = Auth::user()->id;
                $lesson->created_at           = date("Y-m-d H:i:s");
                if($lesson->save()){
                    $meeting                       = new Meeting();
                    $meeting->meeting_id           = $this->random_string(10);
                    $meeting->lesson_id            = $lesson->id;
                    $meeting->user_id              = $lesson->batch->tutor->user->id;
                    $meeting->status               = 'active';
                    $meeting->date                 = $start_date->format("Y-m-d");
                    $meeting->time                 = $start_date->format("H:i");
                    $meeting->created_at           = date("Y-m-d H:i:s");
                    $meeting->save();

                    Session::flash('alert-message','Lesson added successfully.');
                    Session::flash('alert-class','success');
                    return redirect()->route('batches');
                }else{
                    Session::flash('alert-message','Lesson added unsuccessfully.');
                    Session::flash('alert-class','error');
                    return redirect()->route('batches.lesson', $batch_id);
                }
            }
        }catch (\Exception $e) {
            Session::flash('alert-message',$e->getMessage());
            Session::flash('alert-class','error');
            return redirect()->route('batches');
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function lessonTimings($id){
        try {
            $lesson                       = Lessons::where('id',$id)->first();
            $data['page_title']          = 'Update Lesson Timings';
            $data['header']              = 'Update Lesson Timings: '.$lesson->name;
            $data['batch_id']              =  $lesson->batch->id;
            if($lesson){
                $data['lesson']           = $lesson;
                return view('batches.edit_lesson_timing',$data);
            }else{
                return abort(404);
            }
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function storeLessonTimings(Request $request){
        try {

            $lesson_id = $request->id;
            $rules          = [
                "start_date"      			=> "required",
            ];

            $messages       = [
                'start_date.required'           => 'The start date field is required.',
            ];
            $validator      = Validator::make($request->all(), $rules,$messages);
            if ($validator->fails()) {
                return redirect()->route('batches.lesson.timings',$lesson_id)
                                ->withErrors($validator)
                                ->withInput();
            }else{
                $lesson = Lessons::where('id', $lesson_id)->first();
                if($lesson){
                    $batch_id = $lesson->batch->id;
                    $old_start_date = Carbon::createFromTimestamp(strtotime($lesson->start_date));
                    $old_end_date = Carbon::createFromTimestamp(strtotime($lesson->end_date));
                    $minuteDiff = $old_start_date->diffInMinutes($old_end_date);
                    
                    $new_start_date         = Carbon::createFromTimestamp(strtotime($request->start_date));
                    $t                  = $new_start_date->format('Y-m-d')." ".$new_start_date->toTimeString();
                    $new_end_date = Carbon::createFromFormat('Y-m-d H:i:s', $t)->addMinutes($minuteDiff)
                                        ->format('Y-m-d H:i:s');

                    $lesson->start_date = $new_start_date->format('Y-m-d H:i:s');
                    $lesson->end_date = $new_end_date;
                    $lesson->save();

                    $meeting                       = Meeting::where('lesson_id', $lesson->id)->first();
                    if($meeting){
                        $meeting->date                 = date('Y-m-d', strtotime($lesson->start_date));
                        $meeting->time                 = date('H:i', strtotime($lesson->start_date));
                        $meeting->save();
                    }

                    Session::flash('alert-message','Lesson timing updated successfully.');
                    Session::flash('alert-class','success');
                    return redirect()->route('batches.view', $batch_id);
                }
                else{
                    Session::flash('alert-message','Lesson timing updated unsuccessfully.');
                    Session::flash('alert-class','error');
                    return redirect()->route('batches.lesson.timings',$lesson_id);
                }
            }
        }catch (\Exception $e) {
            Session::flash('alert-message',$e->getMessage());
            Session::flash('alert-class','error');
            return redirect()->route('batches.lesson.timings',$lesson_id);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function student($id){
        try {
            $batch                       = Batches::where('id',$id)->first();
            $data['page_title']          = 'Add Student';
            $data['header']              = 'Add Student to Batch '.$batch->name;
            $data['students']           = Students::whereNull('deleted_at')->get();
            if($batch){
                $data['batch']           = $batch;
                return view('batches.add_student',$data);
            }else{
                return abort(404);
            }
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function storeStudent(Request $request){
        try {
            $batch_id = $request->id;
            $rules          = [
                "students"      			=> "required|array|min:1",
            ];

            $messages       = [
                'students.required'                 => 'The days field is required.'
            ];
            $validator      = Validator::make($request->all(), $rules,$messages);
            if ($validator->fails()) {
                return redirect()->route('batches.student',$batch_id)
                                ->withErrors($validator)
                                ->withInput();
            }else{

                $batch = Batches::where('id', $batch_id)->first();
                if($batch->students()->whereIn('students.id', $request->students)->exists()){
                    Session::flash('alert-message','Batch already has students selected.');
                    Session::flash('alert-class','error');
                    return redirect()->route('batches.student', $batch_id);
                }

                $students = $request->students;
                $batch->students()->attach($students);
                Session::flash('alert-message','Student added successfully.');
                Session::flash('alert-class','success');
                return redirect()->route('batches');
            }
        }catch (\Exception $e) {
            Session::flash('alert-message',$e->getMessage());
            Session::flash('alert-class','error');
            return redirect()->route('batches');
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request){
        if ($request->ajax()) {
            try {
                $batch                                   = Batches::where('id',$request->id)->first();
                if($batch){
                    $lessons = $batch->lessons()->get();
                    foreach($lessons as $lesson){
                        $lesson->attendances()->delete();
                        $lesson->meeting()->delete();
                    }

                    $batch->lessons()->delete();
                    $batch->disableLogging();
                    $batch->deleted_by           = Auth::user()->id;
                    $batch->save();
                    $batch->enableLogging();
                    if($batch->delete()){
                        $response['success']    = true;
                        $response['message']    = "Batch deleted successfully.";
                    }else{
                        $response['success']    = false;
                        $response['message']    = "Batch deleted unsuccessfully.";
                    }
                }else{
                    $response['success']                = false;
                    $response['message']                = "Batch record not found.";
                }
            } catch (\Exception $e) {
                $response['success']                    = false;
                $response['message']                    = $e->getMessage();
            }
            return response()->json($response);
        }else{
            return abort(404);
        }
    }
  
    public function feeReminder(Request $request){
        $batch                      = Batches::where('id',$request->id)->first();

        try{
            if($batch){
                if(getSetting('ALLOW_MAIL_FEATURE') == 1){
                    foreach($batch->students()->get() as $student){
                        //TODO: Mail to parent email for fee reminder
                        Mail::to(trim($student->parent_email))->send(new FeeReminderMail($batch, $student));
                    }
                }
            }
        
            Session::flash('alert-message','Feed reminders sent successfully.');
            Session::flash('alert-class','success');
            return redirect()->route("batches.view", $batch->id);
        }
        catch(Exception $e){
            Session::flash('alert-message',$e->getMessage());
            Session::flash('alert-class','error');
            return redirect()->route("batches.view", $batch->id);
        }

    }
}