<?php

namespace App\Http\Controllers;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use App\Mail\PaymentOrderGeneratedMail;
use App\Models\User;
use App\Models\Tutors;
use App\Models\Lessons;
use App\Models\Batches;
use App\Models\PaymentOrders;
use App\Models\PaymentOrderLessons;
use App\Mail\PaymentOrderInvoiceMail;
use Carbon\Carbon;
use DataTables;
use Validator;
use Session;
use Auth;
use Hash;
use Mail;
use PDF;

class PaymentOrderController extends Controller
{

    public function __construct()
    {
        $this->middleware('permission:view-payment-order', ['only' => ['index', 'show']]);
        $this->middleware('permission:create-payment-order', ['only' => ['create', 'store']]);
        $this->middleware('permission:update-payment-order', ['only' => ['edit', 'store']]);
        $this->middleware('permission:delete-payment-order', ['only' => ['destroy']]);
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        try {
            $data = [];
            $data['page_title'] = 'Payment Orders';
            if ($request->ajax()) {
                $orders = null;
                if (Auth::user()->role_name == RoleTutor) {
                    $tutor = Tutors::where('user_id', Auth::user()->id)->first();
                    //print_r(Auth::user()->id);
                    $orders = PaymentOrders::query()->where('tutor_id', $tutor->id)->with('tutor');
                } else {
                    $orders = PaymentOrders::query()->with('tutor');
                }
                //echo "<pre>";
                //print_r($orders);
                //echo "</pre>";
                return DataTables::eloquent($orders)
                    ->addColumn('action', function ($orders) {
                        $action = '';
                        if (Auth::user()->can('view-payment-order')) {
                            $action .= '<a href="' . route("payment-orders.view", $orders->id) . '" title="View Payment Order"><i class="menu-icon tf-icons ti ti-eye"></i></a>&nbsp;';
                        }
                        if (Auth::user()->can('view-payment-order')) {
                            $action .= '<a href="' . route("payment-orders.pdf", $orders->id) . '" title="Download Payment Order"><i class="menu-icon tf-icons ti ti-file-type-pdf"></i></a>&nbsp;';
                        }
                        if (Auth::user()->can('view-payment-order')) {
                            $action .= '<a href="' . route("payment-orders.mail", [$orders->id, $orders->tutor->user_id]) . '" title="Mail Payment Order"><i class="menu-icon tf-icons ti ti-mail"></i></a>&nbsp;';
                        }

                        return $action;
                    })
                    ->addColumn('tutor', function ($orders) {
                        return $orders->tutor->first_name . ' ' . $orders->tutor->last_name;
                    })
                    ->editColumn('status', function ($orders) {
                        $status = '';
                        $type = '';
                        if ($orders->status == PaymentOrderStatusRaised) {
                            $content = "Raised";
                            $type = 'warning';
                        } else if ($orders->status == PaymentOrderStatusPaid) {
                            $content = "Paid";
                            $type = 'success';
                        }
                        return "<span class='btn btn-label-" . $type . " " . $type . "-effect'>" . $content . "</span>";
                    })
                    ->editColumn('start_date', function ($orders) {
                        return ($orders->start_date != null) ? date(config('constant.date_format'), strtotime($orders->start_date)) : '-';
                    })
                    ->editColumn('end_date', function ($orders) {
                        return ($orders->end_date != null) ? date(config('constant.date_format'), strtotime($orders->end_date)) : '-';
                    })
                    ->editColumn('created_at', function ($orders) {
                        return ($orders->created_at != null) ? date(config('constant.date_format'), strtotime($orders->created_at)) : '-';
                    })
                    ->editColumn('updated_at', function ($orders) {
                        return ($orders->updated_at != null) ? date(config('constant.date_format'), strtotime($orders->updated_at)) : '-';
                    })
                    ->filterColumn('tutor_id', function ($query, $keyword) {
                        $query->whereHas('tutor', function ($q) use ($keyword) {
                            $q->where('first_name', 'LIKE', "{$keyword}%")
                                ->where('last_name', 'LIKE', "{$keyword}%");
                        });
                    })
                    ->filterColumn('created_at', function ($query, $keyword) {
                        $query->whereRaw("DATE_FORMAT(created_at,'%d-%m-%Y %h:%i %r') like ?", ["%$keyword%"]);
                    })
                    ->filterColumn('updated_at', function ($query, $keyword) {
                        $query->whereRaw("DATE_FORMAT(updated_at,'%d-%m-%Y %h:%i %r') like ?", ["%$keyword%"]);
                    })
                    ->rawColumns(['status', 'action'])
                    ->make(true);
            }
            return view('payment_orders.index', $data);
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    public function exists(Request $request)
    {
        if (isset($request->id) && $request->id != '') {
            $result = User::where('id', '!=', $request->id)->where('email', $request->email)->count();
        } else {
            $result = User::where('email', $request->email)->count();
        }
        if ($result > 0) {
            return response()->json(false);
        } else {
            return response()->json(true);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        try {
            //print_r("create funtions");die;
            $data['page_title'] = 'Add Payment Order';
            return view('payment_orders.add', $data);
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            $rules = [
                'start_date' => 'required|string|max:255',
                'end_date' => 'required|string|max:255'
            ];

            $messages = [
                'start_date.required' => 'The start date field is required.',
                'end_date.required' => 'The end date field is required.',
            ];
            $validator = Validator::make($request->all(), $rules, $messages);
            if ($validator->fails()) {
                if (isset($request->id) && !empty($request->id)) {
                    return redirect()->route('tutors.edit', $request->id)
                        ->withErrors($validator)
                        ->withInput();
                } else {
                    return redirect()->route('tutors.add')
                        ->withErrors($validator)
                        ->withInput();
                }
            } else {
                $tutors = Tutors::all();
                foreach ($tutors as $tutor) {
                    $lessons = Lessons::with('batch')
                        ->whereHas('batch', function ($query) use ($tutor) {
                            $query->where('tutor_id', $tutor->id);
                        })
                        ->whereIn('status', [LessonStatusCompleted, LessonStatusReportGenerated])
                        ->whereBetween('end_date', [$request->start_date, $request->end_date])->get();
                    $completedLessons = [];

                    foreach ($lessons as $lesson) {
                        if (!PaymentOrderLessons::where('lesson_id', $lesson->id)->exists()) {
                            array_push($completedLessons, $lesson->id);
                        }
                    }
                    if (count($completedLessons) > 0) {
                        $paymentOrder = new PaymentOrders();
                        $paymentOrder->number = generate_po_no();
                        $paymentOrder->tutor_id = $tutor->id;
                        $paymentOrder->start_date = $request->start_date;
                        $paymentOrder->end_date = $request->end_date;
                        $paymentOrder->status = PaymentOrderStatusRaised;
                        $paymentOrder->created_by = Auth::user()->id;
                        $paymentOrder->created_at = date("Y-m-d H:i:s");
                        $paymentOrder->save();

                        foreach ($completedLessons as $id) {
                            $paymentOrderLesson = new PaymentOrderLessons();
                            $paymentOrderLesson->order_id = $paymentOrder->id;
                            $paymentOrderLesson->lesson_id = $id;
                            $paymentOrderLesson->created_by = Auth::user()->id;
                            $paymentOrderLesson->created_at = date("Y-m-d H:i:s");

                            $paymentOrderLesson->save();
                        }

                        // if(getSetting('ALLOW_MAIL_FEATURE') == 1){
                        //     //Mail to Tutor about Payment Order generated
                        //     $user = User::where('id', $tutor->user_id)->first();
                        //     Mail::to(trim($user->email))->send(new PaymentOrderGeneratedMail($user, $paymentOrder));
                        // }

                    }

                }
                Session::flash('alert-message', 'Payment Order added successfully.');
                Session::flash('alert-class', 'success');
                return redirect()->route('payment-orders');
            }
        } catch (\Exception $e) {
            Session::flash('alert-message', $e->getMessage());
            Session::flash('alert-class', 'error');
            return redirect()->route('payment-orders');
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        try {
            $data['page_title'] = 'View Payment Order';
            $paymentOrder = PaymentOrders::where('id', $id)->with(['lessons', 'lessons.lesson', 'lessons.lesson.batch'])->first();

            if ($paymentOrder) {
                $data['paymentOrder'] = $paymentOrder;
                return view('payment_orders.view', $data);
            } else {
                return abort(404);
            }

        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function paid(Request $request)
    {
        try {
            $paymentOrder = PaymentOrders::where('id', $request->order_id)->first();
            $paymentOrder->status = PaymentOrderStatusPaid;
            $paymentOrder->updated_by = Auth::user()->id;
            $paymentOrder->updated_at = date("Y-m-d H:i:s");
            $paymentOrder->paid_at = date("Y-m-d H:i:s");
            $paymentOrder->save();

            $data['paymentOrder'] = $paymentOrder;

            return view('payment_orders.view', $data);

        } catch (\Exception $e) {
            return abort(404);
        }
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request)
    {
        if ($request->ajax()) {
            try {
                $tutor = Tutors::where('id', $request->id)->first();
                if (!is_null($tutor)) {
                    if (Auth::user()->id != $tutor->user->id) {
                        $tutor->disableLogging();
                        $tutor->deleted_by = Auth::user()->id;
                        $tutor->save();
                        $tutor->enableLogging();
                        if ($tutor->delete() && $tutor->user->delete()) {
                            $response['success'] = true;
                            $response['message'] = "Tutor deleted successfully.";
                        } else {
                            $response['success'] = false;
                            $response['message'] = "Tutor deleted unsuccessfully.";
                        }
                    } else {
                        $response['success'] = false;
                        $response['message'] = "Can't delete Tutor because " . $tutor->user()->name . " is active.";
                    }
                } else {
                    $response['success'] = false;
                    $response['message'] = "Tutor record not found.";
                }
            } catch (\Exception $e) {
                $response['success'] = false;
                $response['message'] = $e->getMessage();
            }
            return response()->json($response);
        } else {
            return abort(404);
        }
    }

    public function pdfReport($id)
    {
        $paymentOrder = PaymentOrders::where('id', $id)->with(['lessons', 'lessons.lesson', 'lessons.lesson.batch'])->first();
        $data = [];
        $pdf = PDF::loadView('payment_orders.pdf', compact('paymentOrder', 'data'))->setPaper('a4', 'landscape');
        return $pdf->download('payment_order.pdf');
    }
    public function mailPdfReport($id, $userId)
    {
        $user = User::where('id', $userId)->first();
        //print_r($user->email);die;
        $out = Mail::to(trim($user->email))->send(new PaymentOrderInvoiceMail($id));
        if ($out) {
            Session::flash('alert-message', 'Email sent successfully');
            Session::flash('alert-class', 'success');
            return redirect()->route('payment-orders');
        } else {
            Session::flash('alert-message', 'Email failed to send');
            Session::flash('alert-class', 'error');
            return redirect()->route('payment-orders');
        }
    }
}