<?php

namespace App\Http\Controllers;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use App\Mail\TutorOnBoardMail;
use App\Models\User;
use App\Models\Tutors;
use App\Models\Lessons;
use App\Models\Batches;
use Carbon\Carbon;
use DataTables;
use Validator;
use Session;
use Auth;
use Hash;
use Mail;

class TutorController extends Controller{

    public function __construct(){
        $this->middleware('permission:view-tutor', ['only' => ['index','show']]);
        $this->middleware('permission:create-tutor', ['only' => ['create','store']]);
        $this->middleware('permission:update-tutor', ['only' => ['edit','store']]);
        $this->middleware('permission:delete-tutor', ['only' => ['destroy']]);
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request){
        try {
            $data = [];
            $data['page_title']        = 'Tutors List';
            if($request->ajax()){
                $tutors = Tutors::query()->with('user');
                return DataTables::eloquent($tutors)
                    ->addColumn('action', function($tutors) {
                        $action      = '';
                        if(Auth::user()->can('view-tutor')){
                            $action .= '<a href="'.route("tutors.view",$tutors->id).'" title="View Tutor"><i class="menu-icon tf-icons ti ti-eye"></i></a>&nbsp;';
                        }
                        if(Auth::user()->can('update-tutor')){
                            $action .= '<a href="'.route("tutors.edit",$tutors->id).'" title="Edit Tutor"><i class="menu-icon tf-icons ti ti-edit"></i></a>&nbsp;';
                        }
                        if(Auth::user()->can('delete-tutor')){
                            $action .= '<a href="javascript:void(0);" id="tutor_id_'.$tutors->id.'"  title="Delete Tutor" data-id="'.$tutors->id.'" onclick="deleteRecord(this,'.$tutors->id.');"><i class="menu-icon tf-icons ti ti-trash"></i></a>&nbsp;';
                        }
                    
                        return $action;
                    })
                    ->addColumn('email', function($tutors) {
                        return $tutors->user->email;
                    })
                    ->filterColumn('email', function ($query, $keyword) {
                        $query->whereHas('user', function ($sub_q) use ($keyword) {
                            $sub_q->where('email', 'like', "%$keyword%");
                        });
                    })
                    ->addColumn('name', function($tutors) {
                        return $tutors->first_name.' '.$tutors->last_name;
                    })
                    ->filterColumn('name', function ($query, $keyword) {
                        $query->where('first_name', 'LIKE', "{$keyword}%")
                            ->orWhere('last_name', 'LIKE', "{$keyword}%");
                    })
                    ->editColumn('status', function($tutors) {
                        $status = '';
                        $type = '';
                        if($tutors->status == 0) {
                            $content = "PENDING";
                            $type = 'warning';
                        }
                        else if($tutors->status == 1) {
                            $content = "APPROVED";
                            $type = 'success';
                        }
                        else if($tutors->status == 2) {
                            $content = "REJECTED";
                            $type = 'danger';
                        }
                        else if($tutors->status == 3) {
                            $content = "DORMENT";
                            $type = 'danger';
                        }
                        return "<span class='btn btn-label-".$type." ".$type."-effect'>".$content."</span>";
                    })
                    ->editColumn('created_at', function($users) {
                        return ($users->created_at!=null)?date(config('constant.date_format'), strtotime($users->created_at)):'-';
                    })
                    ->editColumn('updated_at', function($users) {
                        return ($users->updated_at!=null)?date(config('constant.date_format'), strtotime($users->updated_at)):'-';
                    })
                    ->filterColumn('created_at', function ($query, $keyword) {
                        $query->whereRaw("DATE_FORMAT(created_at,'%d-%m-%Y %h:%i %r') like ?", ["%$keyword%"]);
                    })
                    ->filterColumn('updated_at', function ($query, $keyword) {
                        $query->whereRaw("DATE_FORMAT(updated_at,'%d-%m-%Y %h:%i %r') like ?", ["%$keyword%"]);
                    })
                    ->rawColumns(['status', 'action'])
                    ->make(true);
            }
            return view('tutors.index',$data);
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    public function exists(Request $request){
        if(isset($request->id) && $request->id!=''){
            $result = User::where('id', '!=' , $request->id)->where('email',$request->email)->count();
        }else{
            $result = User::where('email',$request->email)->count();
        }
        if($result>0){
            return response()->json(false);
        }else{
            return response()->json(true);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(){
        try {
            $data['page_title']         = 'Add Tutor';
            $data['roles']              = Role::whereNull('deleted_at')->get();
            return view('tutors.add',$data);
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request){
        try {
            $rules          = [
                'first_name'              => 'required|string|max:255',
                'last_name'              => 'required|string|max:255',
                'status'                  => 'required',
            ];

            if (!isset($request->id)) {
                $rules['email']         = 'required|string|email|max:255|unique:users';
                $rules['password']         = 'required|string|min:8';
            }

            $messages       = [
                'first_name.required'         => 'The first name field is required.',
                'last_name.required'         => 'The last name field is required.',
                'email.required'        => 'The email field is required.',
                'status.required'        => 'The status field is required.',
                'password.required' 	=> 'The password field is required.',
            ];
            $validator      = Validator::make($request->all(), $rules,$messages);
            if ($validator->fails()) {
                if (isset($request->id) && !empty($request->id)) {
                    return redirect()->route('tutors.edit',$request->id)
                                ->withErrors($validator)
                                ->withInput();
                }else{
                    return redirect()->route('tutors.add')
                                ->withErrors($validator)
                                ->withInput();
                }
            }else{
                $name                       = $request->first_name." ".$request->last_name;

                $role = Role::where('name', 'tutor')->whereNull('deleted_at')->first();

                if (isset($request->id) && !empty($request->id)) {
                    $id                     = $request->id;
                    $tutor                  = Tutors::where('id',$id)->first();
                    $user                   = $tutor->user;
                    $user->name             = trim($name);
                    if($request->email != $user->email){
                        $user->email            = trim($request->email);
                    }
                    $user->updated_by       = Auth::user()->id;
                    $user->updated_at       = date("Y-m-d H:i:s");
                    if($user->save()){
                        $user->syncRoles($role);

                        $tutor->status                       = $request->status;
                        $tutor->first_name                   = $request->first_name;
                        $tutor->last_name                    = $request->last_name;
                        $tutor->user_id                   = $user->id;
                        $tutor->updated_by                   = Auth::user()->id;
                        $tutor->updated_at                   = date("Y-m-d H:i:s");
                        $tutor->save();
                        
                        Session::flash('alert-message','Tutor updated successfully.');
                        Session::flash('alert-class','success');
                        return redirect()->route('tutors');
                    }else{
                        Session::flash('alert-message','Tutor updated unsuccessfully.');
                        Session::flash('alert-class','error');
                        return redirect()->route('tutors.edit',$id);
                    }
                }else{
                    $user                       = new User();
                    $user->name                 = trim($name);
                    $password                   = $request->password;
                    $user->password             = bcrypt($password);
                    $user->email                = trim($request->email);
                    $user->email_verified_at    = date("Y-m-d H:i:s");
                    $user->created_by           = Auth::user()->id;
                    $user->created_at           = date("Y-m-d H:i:s");
                    if($user->save()){
                        $user->assignRole($role);

                        $tutor                               = new Tutors;
                        $tutor->first_name                   = $request->first_name;
                        $tutor->last_name                    = $request->last_name;
                        $tutor->user_id                      = $user->id;
                        $tutor->status                       = $request->status;
                        $tutor->created_by                   = Auth::user()->id;
                        $tutor->created_at                   = date("Y-m-d H:i:s");
                        $tutor->save();

                        if(getSetting('ALLOW_MAIL_FEATURE') == 1){
                            //Mail to Tutor
                            Mail::to(trim($request->email))->send(new TutorOnBoardMail($user, $password));
                        }
                        
                        
                        Session::flash('alert-message','Tutor added successfully.');
                        Session::flash('alert-class','success');
                        return redirect()->route('tutors');
                    }else{
                        Session::flash('alert-message','Tutor added unsuccessfully.');
                        Session::flash('alert-class','error');
                        return redirect()->route('tutors.add');
                    }
                }
            }
        }catch (\Exception $e) {
            Session::flash('alert-message',$e->getMessage());
            Session::flash('alert-class','error');
            return redirect()->route('tutors');
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id){
        try {
            $data['page_title']         = 'View Tutor';
            $tutor                      = Tutors::where('id',$id)->first();
            if($tutor){
                //$data['user']   = $user;
                $data['tutor']   = $tutor;
                return view('tutors.view',$data);
            }else{
                return abort(404);
            }

        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id){
        try {
            $data['page_title']         = 'Edit Tutor';
            $tutor                       = Tutors::where('id',$id)->first();
            if($tutor){
                $data['roles']           = Role::whereNull('deleted_at')->get();
                $data['tutor']           = $tutor;
                return view('tutors.edit',$data);
            }else{
                return abort(404);
            }
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request){
        if ($request->ajax()) {
            try {
                $tutor                                   = Tutors::where('id',$request->id)->first();
                if(!is_null($tutor)){
                    $tutor->disableLogging();
                    $tutor->deleted_by           = Auth::user()->id;
                    $tutor->status = 3;
                    $tutor->save();
                    //$tutor->enableLogging();
                    if($tutor->save()){
                        if($tutor->user)  {
                            $tutor->user->deleted_by = Auth::user()->id;
                            $tutor->user->save();
                        }
                        $response['success']    = true;
                        $response['message']    = "Tutor deleted successfully.";
                    }else{
                        $response['success']    = false;
                        $response['message']    = "Tutor deleted unsuccessfully.";
                    }
                }else{
                    $response['success']                = false;
                    $response['message']                = "Tutor record not found.";
                }
            } catch (\Exception $e) {
                $response['success']                    = false;
                $response['message']                    = $e->getMessage();
            }
            return response()->json($response);
        }else{
            return abort(404);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function lessons(Request $request){
        try {
            $data['page_title']         = 'My Lessons';
            $data['lessons']            = [];
            if(Auth::user()->role_name == RoleTutor){
                $tutor = Tutors::where('user_id', Auth::user()->id)->first();
                $lessons = collect([]);
                foreach($tutor->batches as $batch)
                    $lessons = $lessons->merge($batch->lessons);
                
                $data['lessons']             = $lessons;
            }
            else{
                $data['lessons']             = Lessons::whereNull('deleted_at')->get();
            }

            return view('tutors.lessons',$data);
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    public function batches(Request $request){
        try {
            if($request->ajax()){
                $batches = Batches::where('tutor_id', $request->id)->whereNull('deleted_at');
                
                return DataTables::eloquent($batches)
                    ->addColumn('action', function($batches) {
                        $action      = '';
                        if(Auth::user()->can('view-batch')){
                            $action .= '<a href="'.route("batches.view",$batches->id).'"><i class="menu-icon tf-icons ti ti-eye"></i></a>&nbsp;';
                        }
                    
                        return $action;
                    })
                    ->addColumn('tutor_id', function($batches) {
                        return $batches->tutor->first_name." ".$batches->tutor->last_name;
                    })
                    ->filterColumn('tutor_id', function ($query, $keyword) {
                        $query->whereHas('tutor', function ($q) use ($keyword) {
                            $q->where('first_name', 'LIKE', "{$keyword}%")
                            ->where('last_name', 'LIKE', "{$keyword}%");
                        });
                    })
                    ->addColumn('lessons', function($batches) {
                        return $batches->lessons->count();
                    })
                    ->editColumn('created_at', function($users) {
                        return ($users->created_at!=null)?date(config('constant.date_format'), strtotime($users->created_at)):'-';
                    })
                    ->editColumn('updated_at', function($users) {
                        return ($users->updated_at!=null)?date(config('constant.date_format'), strtotime($users->updated_at)):'-';
                    })
                    ->filterColumn('created_at', function ($query, $keyword) {
                        $query->whereRaw("DATE_FORMAT(created_at,'%d-%m-%Y %h:%i %r') like ?", ["%$keyword%"]);
                    })
                    ->filterColumn('updated_at', function ($query, $keyword) {
                        $query->whereRaw("DATE_FORMAT(updated_at,'%d-%m-%Y %h:%i %r') like ?", ["%$keyword%"]);
                    })
                    ->rawColumns([])
                    ->make(true);
            }
        }catch (\Exception $e) {
            return abort(404);
        }
    }

    public function schedule(Request $request){
        try {
            $data['page_title']         = 'Schedule';
            $lessons                    = [];
            if(Auth::user()->role_name == RoleTutor){
                $tutor = Tutors::where('user_id', Auth::user()->id)->first();
                $lessons = Lessons::whereHas('batch', function($q) use($tutor){
                                $q->where('tutor_id', $tutor->id);
                            })->whereNull('deleted_at')
                            ->get()
                            ->map(function ($l) {
                                $lesson['id']              = $l->id;
                                $lesson['title']           = $l->batch->name.'-'.$l->name;
                                $lesson['url']             = route('lessons.view', $l->id);
                                $lesson['start']           = Carbon::parse($l->start_date)->format('Y-m-d H:i:s');
                                $lesson['end']             = Carbon::parse($l->end_date)->format('Y-m-d H:i:s');
                                $lesson['allDay']          = false;
                                
                                
                                
                                $lesson['extendedProps']   = [];

                                if($l->status == LessonStatusPlanned){
                                    $lesson['extendedProps']['calendar']   = 'planned';
                                }
                                else if($l->status == LessonStatusInProgress){
                                    $lesson['extendedProps']['calendar']   = 'in_progress';
                                }
                                else if($l->status == LessonStatusCompleted){
                                    $lesson['extendedProps']['calendar']   = 'completed';
                                }
                                else if($l->status == LessonStatusCancelled){
                                    $lesson['extendedProps']['calendar']   = 'cancelled';
                                }
                                else if($l->status == LessonStatusReportGenerated){
                                    $lesson['extendedProps']['calendar']   = 'report_generated';
                                }

                                return $lesson;
                            });
            }
            else{
                $lessons = Lessons::whereNull('deleted_at')
                                    ->get()
                                    ->map(function ($l) {
                                        $lesson['id']              = $l->id;
                                        $lesson['title']           = $l->batch->name.'-'.$l->name;
                                        $lesson['url']             = route('lessons.view', $l->id);
                                        $lesson['start']           = Carbon::parse($l->start_date)->format('Y-m-d H:i:s');
                                        $lesson['end']             = Carbon::parse($l->end_date)->format('Y-m-d H:i:s');
                                        $lesson['allDay']          = false;
                                        
                                        $lesson['extendedProps']   = [];
        
                                        if($l->status == LessonStatusPlanned){
                                            $lesson['extendedProps']['calendar']   = 'planned';
                                        }
                                        else if($l->status == LessonStatusInProgress){
                                            $lesson['extendedProps']['calendar']   = 'in_progress';
                                        }
                                        else if($l->status == LessonStatusCompleted){
                                            $lesson['extendedProps']['calendar']   = 'completed';
                                        }
                                        else if($l->status == LessonStatusCancelled){
                                            $lesson['extendedProps']['calendar']   = 'cancelled';
                                        }
                                        else if($l->status == LessonStatusReportGenerated){
                                            $lesson['extendedProps']['calendar']   = 'report_generated';
                                        }
        
                                        return $lesson;
                                    });
            }
            
            $data['lessons']         = $lessons->toJSON();
            return view('tutors.schedule',$data);
        } catch (\Exception $e) {
            return abort(404);
        }
    }
}
