<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Http\Requests\Auth\LoginRequest;
use App\Providers\RouteServiceProvider;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\View\View;
use App\Models\User;
use App\Models\MeetingParticipants;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Session;
use Validator;


class AuthenticatedSessionController extends Controller
{
    /**
     * Display the login view.
     */
    public function create(): View
    {
        if(Auth::check()){
            return view('home');
        }

        return view('auth.login');
    }

    /**
     * Handle an incoming authentication request.
     */
    public function store(LoginRequest $request) : RedirectResponse
    {
        try{

            $rules          = [
                'email'                  => 'required',
                'password'               => 'required',
            ];

            $messages       = [
                'email.required'                 => 'The email field is required.',
                'password.required'          => 'The password field is required.',
            ];

            $validator      = Validator::make($request->all(), $rules,$messages);

            if ($validator->fails()) {
                return redirect()->route('login')
                                ->withErrors($validator)
                                ->withInput();
            }
            else{
                $email = $request->email;
                $password = $request->password;
                $remember = $request->remember == "on";

                $user = User::where('email', $request->email)->first();

                if($user){
                    if (Auth::attempt([ 'email'=> $email, 'password'=> $password ], $remember)) {
                        if(MeetingParticipants::where('user_id', $user->id)->exists()){
                            MeetingParticipants::where('user_id', $user->id)->delete();
                            Session::flash('alert-message','Your current meetings sessions has been dismissed.');
                            Session::flash('alert-class','warning');
                        }
                        
                        if(Auth::user()->role_name == RoleStudent){
                            return redirect()->intended(RouteServiceProvider::BATCHES);
                        }
                        else{
                            return redirect()->intended(RouteServiceProvider::HOME);
                        }
                    }
                    else{
                        Session::flash('alert-message','Incorrect email or password.');
                        Session::flash('alert-class','error');
                        return redirect()->intended(RouteServiceProvider::LOGIN);
                    }
                }
                else{
                    Session::flash('alert-message','Email does not exists.');
                    Session::flash('alert-class','error');
                    return redirect()->intended(RouteServiceProvider::LOGIN);
                }
            }
        }
        catch (\Exception $e) {
            Session::flash('alert-message',$e->getMessage());
            Session::flash('alert-class','error');
            return redirect()->intended(RouteServiceProvider::LOGIN);
        }

        return redirect()->intended(RouteServiceProvider::LOGIN);
    }

    /**
     * Destroy an authenticated session.
     */
    public function destroy(Request $request): RedirectResponse
    {
        Auth::guard('web')->logout();

        $request->session()->invalidate();

        $request->session()->regenerateToken();

        return redirect('/');
    }
}
