<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use App\Models\Permissions;
use DataTables;
use Validator;
use Session;

class PermissionController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:view-permission', ['only' => ['index']]);
        $this->middleware('permission:create-permission', ['only' => ['create','store']]);
        $this->middleware('permission:update-permission', ['only' => ['update','edit']]);
        $this->middleware('permission:delete-permission', ['only' => ['destroy']]);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request){
        try {
            $data = [];
            $data['page_title']         = 'Permission List';
            if($request->ajax()){
                $permissions = Permission::query()->whereNull('deleted_at');
                return Datatables::eloquent($permissions)
                    ->addColumn('action', function($permissions) {
                        $action      = '';
                        if(Auth::user()->can('view-permission')){
                            $action .= '<a href="'.route("permissions.view",$permissions->id).'" title="View Permission"><i class="menu-icon tf-icons ti ti-eye"></i></a>&nbsp;';
                        }
                        if(Auth::user()->can('update-permission')){
                            $action .= '<a href="'.route("permissions.edit",$permissions->id).'" title="Edit Permission"><i class="menu-icon tf-icons ti ti-edit"></i></a>&nbsp;';
                        }
                        if(Auth::user()->can('delete-permission')){
                            $action .= '<a href="javascript:void(0);" id="permission_id_'.$permissions->id.'" title="Delete Permission" data-id="'.$permissions->id.'" onclick="deleteRecord(this,'.$permissions->id.');"><i class="menu-icon tf-icons ti ti-trash"></i></a>&nbsp;';
                        }
                        return $action;
                    })
                    ->editColumn('created_at', function($permissions) {
                        return ($permissions->created_at!=null)?date('d-m-Y', strtotime($permissions->created_at)):'-';
                    })
                    ->editColumn('updated_at', function($permissions) {
                        return ($permissions->updated_at!=null)?date('d-m-Y', strtotime($permissions->updated_at)):'-';
                    })
                    ->filterColumn('created_at', function ($query, $keyword) {
                        $query->whereRaw("DATE_FORMAT(created_at,'%d-%m-%Y') like ?", ["%$keyword%"]);
                    })
                    ->filterColumn('updated_at', function ($query, $keyword) {
                        $query->whereRaw("DATE_FORMAT(updated_at,'%d-%m-%Y') like ?", ["%$keyword%"]);
                    })
                    ->rawColumns(['action'])
                    ->make(true);
            }
            return view('permissions.index',$data);
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    public function exists(Request $request){
        $slug       = preg_replace('/[^(\x20-\x7F)\x0A\x0D]*/','', $request->name);
        $slug       = Str::slug($slug, "-");
        if(isset($request->id) && $request->id!=''){
            $result = Permissions::where('id', '!=' , $request->id)->where('name',$slug)->where('guard_name','web')->count();
        }else{
            $result = Permissions::where('name',$slug)->where('guard_name','web')->count();
        }
        if($result>0){
            return response()->json(false);
        }else{
            return response()->json(true);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(){
        try {
            $data['page_title']         = 'Add Permission';
            return view('permissions.add',$data);
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request){
        try {
            $rules          = [
                'name'                      => 'required|string|max:255'
            ];
            $messages       = [
                'name.required'             => 'The name field is required.',
            ];
            $validator      = Validator::make($request->all(), $rules,$messages);
            if ($validator->fails()) {
                return redirect()->route('permissions.add')
                            ->withErrors($validator)
                            ->withInput();
            }else{
                $slug                       =   preg_replace('/[^(\x20-\x7F)\x0A\x0D]*/','', $request->name);
                $slug                       =   Str::slug($slug, "-");
                $permission                 =   Permission::create([
                    'guard_name'            =>  'web',
                    'name'                  =>  $slug,
                    'display_name'          =>  $request->name,
                    'description'           =>  $request->description,
                    'created_by'            =>  Auth::user()->id,
                    'created_at'            =>  date("Y-m-d H:i:s"),
                    'updated_at'            =>  null
                ]);
                if($permission){
                    $userModel              =   Auth::user();
                    $permissionModel        =   Permissions::where('id',$permission->id)->first();
                    $properties             =   $permissionModel->toArray();
                    unset($properties['id']);
                    activity('Permission')->performedOn($permissionModel)->causedBy($userModel)->withProperties(['attributes' => $properties])->log('created');
                    // $role                   =   Role::findByName('admin');
                    // if(!is_null($role)){
                    //     $role->givePermissionTo($permission->name);
                    // }
                    Session::flash('alert-message','Permission added successfully.');
                    Session::flash('alert-class','success');
                    return redirect()->route('permissions');
                }else{
                    Session::flash('alert-message','Permission added unsuccessfully.');
                    Session::flash('alert-class','error');
                    return redirect()->route('permissions.add');
                }
            }
        }catch (\Exception $e) {
            Session::flash('alert-message',$e->getMessage());
            Session::flash('alert-class','error');
            return redirect()->route('permissions.add');
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id){
        try {
            $data['page_title']         = 'View Permission';
            $permission                 = Permissions::where('id',$id)->whereNull('deleted_at')->first();
            if($permission){
                $data['permission']     = $permission;
                return view('permissions.view',$data);
            }else{
                return abort(404);
            }
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id){
        try {
            $data['page_title']         = 'Edit Permission';
            $permission                 = Permissions::where('id',$id)->whereNull('deleted_at')->first();
            if($permission){
                $data['permission']     = $permission;
                return view('permissions.edit',$data);
            }else{
                return abort(404);
            }
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id){
        try {
            $rules          = [
                'name'                  => 'required|string|max:255'
            ];
            $messages       = [
                'name.required'         => 'The name field is required.'
            ];
            $validator      = Validator::make($request->all(), $rules,$messages);
            if ($validator->fails()) {
                return redirect()->route('permissions.edit',$id)
                            ->withErrors($validator)
                            ->withInput();
            }else{
                $slug                           =   preg_replace('/[^(\x20-\x7F)\x0A\x0D]*/','', $request->name);
                $slug                           =   Str::slug($slug, "-");
                $permission                     =   Permission::findById($id);
                $properties                     =   $permission->toArray();
                foreach ($properties as $key => $value) {
                    if(!(in_array($key, ['name','display_name','updated_at','updated_by']))){
                        unset($properties[$key]);
                    }
                }
                if($permission){
                    $permission->name               =   $slug;
                    $permission->display_name       =   $request->name;
                    $permission->description        =   $request->description;
                    $permission->updated_by         =   Auth::user()->id;
                    $permission->updated_at         =   date("Y-m-d H:i:s");
                    if($permission->save()){
                        $userModel                  =   Auth::user();
                        $permissionModel            =   Permissions::where('id',$permission->id)->first();
                        $updatedproperties          =   $permissionModel->toArray();
                        foreach ($updatedproperties as $key => $value) {
                            if(!(in_array($key, ['name','display_name','updated_at','updated_by']))){
                                unset($updatedproperties[$key]);
                            }
                        }
                        activity('Permission')->performedOn($permissionModel)->causedBy($userModel)->withProperties(['attributes' => $updatedproperties,'old'=>$properties])->log('updated');
                        Session::flash('alert-message','Permission updated successfully.');
                        Session::flash('alert-class','success');
                        return redirect()->route('permissions');
                    }else{
                        Session::flash('alert-message','Permission updated unsuccessfully.');
                        Session::flash('alert-class','error');
                        return redirect()->route('permissions.edit',$id);
                    }
                }else{
                    Session::flash('alert-message','Permission record not found.');
                    Session::flash('alert-class','error');
                    return redirect()->route('permissions');
                }
            }
        }catch (\Exception $e) {
            Session::flash('alert-message',$e->getMessage());
            Session::flash('alert-class','error');
            return redirect()->route('permissions.edit',$id);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request){
        if ($request->ajax()) {
            try {
                $permission                         = Permission::findById($request->id);
                if(!is_null($permission)){
                    $roles                          = Role::all();
                    if(!is_null($roles)){
                        foreach ($roles as $key => $role) {
                            if($role->hasPermissionTo($permission->name)){
                                $role->revokePermissionTo($permission->name);
                            }
                        }
                    }
                    $permissions                    = Permissions::where('id',$request->id)->first();
                    $permissions->disableLogging();
                    $permissions->deleted_by        = Auth::user()->id;
                    $permissions->save();
                    $permissions->enableLogging();
                    if($permissions->delete()){
                        $response['success']        = true;
                        $response['message']        = "Permission deleted successfully.";
                    }else{
                        $response['success']        = false;
                        $response['message']        = "Permission deleted unsuccessfully.";
                    }
                }else{
                    $response['success']            = false;
                    $response['message']            = "Permission record not found.";
                }
            } catch (\Exception $e) {
                $response['success']                = false;
                $response['message']                = $e->getMessage();
            }
            return response()->json($response);
        }else{
            return abort(404);
        }
    }
}