<?php

namespace App\Http\Controllers;

use App\Http\Requests\ProfileUpdateRequest;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\View\View;
use App\Mail\UserPasswordChangeMail;
use App\Models\User;
use Session;
use Mail;

class ProfileController extends Controller
{
	public function showProfile(Request $request){
	    try {
	    	$data 				= [];
	    	$data['page_title'] = 'Profile';
	        return view('profile.edit',$data);
	    } catch (\Exception $e) {
            return abort(404);
	    }
	}

	public function profileUpdate(Request $request){
        try {
            $userId     = Auth::id();
            $rules      = [
                'name'   => 'required',
                'email'  => 'required|string|email|max:255|unique:users,email,'.$userId
            ];
            $messages   = [
                'name.required'  => 'The fullname field is required.',
                'email.required' => 'The email field is required.'
            ];
            $validator = Validator::make($request->all(), $rules, $messages);
            if ($validator->fails()) {
                return redirect()->route('profile')
                    ->withErrors($validator)
                    ->withInput();
            } else {
                $user = User::where('id',$userId)->first();
                if($user){
                    $user->name  = $request->name;
                    $user->email = $request->email;
                    $user->save();
                    Session::flash('alert-message', 'Profile updated successfully.');
                    Session::flash('alert-class', 'success');
                    return redirect()->route('profile');
                }else{
                    Session::flash('alert-message', 'User record not found.');
                    Session::flash('alert-class', 'error');
                    return redirect()->route('profile');
                }
            }
        } catch (\Exception $e) {
            Session::flash('alert-message', $e->getMessage());
            Session::flash('alert-class', 'error');
            return redirect()->route('profile');
        }
	}

	public function showPassword(Request $request){
	    try {
	    	$data 				= [];
	    	$data['page_title'] = 'Change Password';
	        return view('profile.password',$data);
	    } catch (\Exception $e) {
            return abort(404);
	    }
	}

	public function passwordUpdate(Request $request){
		try {
            $userId     = Auth::id();
            $rules 	    = [
                'current_password' 					=> 'required|string|min:8',
                'password' 							=> 'required|string|min:8|confirmed',
                'password_confirmation'				=> 'required|string|min:8'
            ];
            $messages   = [
                'current_password.required' 		=> 'The current password field is required.',
                'password.required' 				=> 'The new password field is required.',
                'password_confirmation.required' 	=> 'The confirm password field is required.'
            ];
            $validator = Validator::make($request->all(), $rules, $messages);
            if ($validator->fails()) {
                return redirect()->route('profile')
                    ->withErrors($validator)
                    ->withInput();
            } else {
                $user = User::where('id',$userId)->first();
                if($user){
                    if(Hash::check($request->current_password, Auth::user()->password)){
                        $new_password = $request->password;
                        $user->password = bcrypt($request->password);
                        $user->save();
                        
                        if(getSetting('ALLOW_MAIL_FEATURE') == 1){
                            Mail::to(trim($user->email))->send(new UserPasswordChangeMail($user, $new_password));
                        }
                        
                        Auth::logout();
                        Session::flash('alert-message', "Password changed successfully. Please login bellow with new password.");
                        Session::flash('alert-class', "success");
                        return redirect('/');
                    }else{
                        Session::flash('alert-message', 'Please enter correct current password.');
                        Session::flash('alert-class', 'error');
                        return redirect()->route('profile');
                    }
                }else{
                    Session::flash('alert-message', 'User record not found.');
                    Session::flash('alert-class', 'error');
                    return redirect()->route('profile');
                }
            }
        } catch (\Exception $e) {
            Session::flash('alert-message', $e->getMessage());
            Session::flash('alert-class', 'error');
            return redirect()->route('profile');
        }
	}
    

    /**
     * Display the user's profile form.
     */
    public function edit(Request $request): View
    {
        return view('profile.edit', [
            'user' => $request->user(),
        ]);
    }

    /**
     * Update the user's profile information.
     */
    public function update(ProfileUpdateRequest $request): RedirectResponse
    {
        $request->user()->fill($request->validated());

        if ($request->user()->isDirty('email')) {
            $request->user()->email_verified_at = null;
        }

        $request->user()->save();

        return Redirect::route('profile.edit')->with('status', 'profile-updated');
    }

    /**
     * Delete the user's account.
     */
    public function destroy(Request $request): RedirectResponse
    {
        $request->validateWithBag('userDeletion', [
            'password' => ['required', 'current_password'],
        ]);

        $user = $request->user();

        Auth::logout();

        $user->delete();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return Redirect::to('/');
    }
}
