<?php

namespace  App\Http\Controllers;

use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use App\Models\Roles;
use DataTables;
use Validator;
use Session;

class RoleController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:view-role', ['only' => ['index']]);
        $this->middleware('permission:create-role', ['only' => ['create','store','addPermissionToRole','givePermissionToRole']]);
        $this->middleware('permission:update-role', ['only' => ['update','edit']]);
        $this->middleware('permission:delete-role', ['only' => ['destroy']]);
    }
    
    public function index(Request $request)
    {
        $data['page_title']         = 'Edit Role';
        if($request->ajax()){
            $roles = Role::query()->whereNull('deleted_at');
            return Datatables::eloquent($roles)
            ->addColumn('action', function($roles) {
                $action      = '';
                if(Auth::user()->can('view-role')){
                    $action .= '<a href="'.route("roles.view",$roles->id).'" title="View Role"><i class="menu-icon tf-icons ti ti-eye"></i></a>&nbsp;';
                }
                if(Auth::user()->can('update-role')){
                    $action .= '<a href="'.route("roles.edit",$roles->id).'" title="Edit Role"><i class="menu-icon tf-icons ti ti-edit"></i></a>&nbsp;';
                }
                if(Auth::user()->can('delete-role')){
                    $action .= '<a href="javascript:void(0);" id="role_id_'.$roles->id.'" title="Delete Role" data-id="'.$roles->id.'" onclick="deleteRecord(this,'.$roles->id.');"><i class="menu-icon tf-icons ti ti-trash"></i></a>&nbsp;';
                }
                return $action;
            })
            ->editColumn('created_at', function($roles) {
                return ($roles->created_at!=null)?date(config('constant.date_format'), strtotime($roles->created_at)):'-';
            })
            ->editColumn('updated_at', function($roles) {
                return ($roles->updated_at!=null)?date(config('constant.date_format'), strtotime($roles->updated_at)):'-';
            })
            ->filterColumn('created_at', function ($query, $keyword) {
                $query->whereRaw("DATE_FORMAT(created_at,'%d-%m-%Y %h:%i %r') like ?", ["%$keyword%"]);
            })
            ->filterColumn('updated_at', function ($query, $keyword) {
                $query->whereRaw("DATE_FORMAT(updated_at,'%d-%m-%Y %h:%i %r') like ?", ["%$keyword%"]);
            })
            ->rawColumns(['action'])
            ->make(true);
        }
        return view('roles.index', $data);
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id){
        try {
            $data['page_title']         = 'View Role';
            $role                       = Role::where('id',$id)->whereNull('deleted_at')->first();
            if($role){
                $data['role']           = $role;
                $data['permissions']    = Permission::whereNull('deleted_at')->get();
                return view('roles.view',$data);
            }else{
                return abort(404);
            }
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(){
        try {
            $data['page_title']         = 'Add Role';
            $data['permissions']        = Permission::query()->whereNull('deleted_at')->get();
            return view('roles.add',$data);
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request){
        try {
            $rules          = [
                'name'                      => 'required|string|max:255',
                "permissions"    			=> "required|array|min:1",
            ];
            $messages       = [
                'name.required'             => 'The name field is required.',
                'permissions.required'      => 'Please select at least 1 permission.'
            ];
            $validator      =   Validator::make($request->all(), $rules,$messages);
            
            if ($validator->fails()) {
                return redirect()->route('roles.add')
                            ->withErrors($validator)
                            ->withInput();
            }else{
                $slug       =   preg_replace('/[^(\x20-\x7F)\x0A\x0D]*/','', $request->name);
                $slug       =   Str::slug($slug, "-");
                $role       =   Role::create([
                    'name'          =>  $slug,
                    'display_name'  =>  $request->name,
                    'description'   =>  $request->description,
                    'created_by'    =>  Auth::user()->id,
                    'created_at'    =>  date("Y-m-d H:i:s"),
                    'updated_at'    =>  null
                ]);
                
                if(!is_null($role)){
                    $userModel              =   Auth::user();
                    $roleModel              =   Roles::where('id',$role->id)->first();
                    $properties             =   $roleModel->toArray();
                    unset($properties['id']);
                    activity('Role')->performedOn($roleModel)->causedBy($userModel)->withProperties(['attributes' => $updatedproperties,'old'=>$properties])->log('updated');
                    if(count($request->permissions)>0){
                        foreach ($request->permissions as $key => $permission) {
                            $role->givePermissionTo($permission);
                        }
                    }
                    Session::flash('alert-message','Role added successfully.');
                    Session::flash('alert-class','success');
                    return redirect()->route('roles');
                }else{
                    Session::flash('alert-message','Role added unsuccessfully.');
                    Session::flash('alert-class','error');
                    return redirect()->route('roles.add');
                }

            }
        }catch (\Exception $e) {
            Session::flash('alert-message',$e->getMessage());
            Session::flash('alert-class','error');
            return redirect()->route('roles.add');
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id){
        try {
            $data['page_title']         = 'Edit Role';
            $role                       = Role::where('id',$id)->whereNull('deleted_at')->first();
            if($role){
                $data['role']           = $role;
                $data['permissions']    = Permission::whereNull('deleted_at')->get();
                return view('roles.edit',$data);
            }else{
                return abort(404);
            }
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id){
        try {
            $rules          = [
                'name'                      => 'required|string|max:255',
                "permissions"               => "required|array|min:1",
            ];
            $messages       = [
                'name.required'             => 'The name field is required.',
                'permissions.required'      => 'Please select at least 1 permission.'
            ];
            $validator      = Validator::make($request->all(), $rules,$messages);
            if ($validator->fails()) {
                return redirect()->route('permissions.edit',$id)
                            ->withErrors($validator)
                            ->withInput();
            }else{
                $role                       = Role::findById($id);
                $properties                 = $role->toArray();
                foreach ($properties as $key => $value) {
                    if(!(in_array($key, ['name', 'description','updated_at','updated_by']))){
                        unset($properties[$key]);
                    }
                }
                $slug                       = preg_replace('/[^(\x20-\x7F)\x0A\x0D]*/','', $request->name);
                $slug                       = Str::slug($slug, "-");
                $role->display_name         = $request->name;
                $role->description          = $request->description;
                $role->updated_by           = Auth::user()->id;
                $role->updated_at           = date("Y-m-d H:i:s");
                $role->save();
                if(!is_null($role)){
                    $userModel              =   Auth::user();
                    $roleModel              =   Roles::where('id',$role->id)->first();
                    $updatedproperties      =   $roleModel->toArray();
                    foreach ($updatedproperties as $key => $value) {
                        if(!(in_array($key, ['name','description','updated_at','updated_by']))){
                            unset($updatedproperties[$key]);
                        }
                    }
                    activity('Role')->performedOn($roleModel)->causedBy($userModel)->withProperties(['attributes' => $updatedproperties,'old'=>$properties])->log('updated');
                    $role->syncPermissions($request->permissions);
                    Session::flash('alert-message','Role updated successfully.');
                    Session::flash('alert-class','success');
                    return redirect()->route('roles');
                }else{
                    Session::flash('alert-message','Role updated unsuccessfully.');
                    Session::flash('alert-class','error');
                    return redirect()->route('roles.edit',$id);
                }
            }
        }catch (\Exception $e) {
            Session::flash('alert-message',$e->getMessage());
            Session::flash('alert-class','error');
            return redirect()->route('roles.edit',$id);
        }
    }


    public function destroy($roleId)
    {
        $role = Role::find($roleId);
        $role->delete();
        return redirect('roles')->with('status','Role Deleted Successfully');
    }

    public function addPermissionToRole($roleId)
    {
        $permissions = Permission::get();
        $role = Role::findOrFail($roleId);
        $rolePermissions = DB::table('role_has_permissions')
                                ->where('role_has_permissions.role_id', $role->id)
                                ->pluck('role_has_permissions.permission_id','role_has_permissions.permission_id')
                                ->all();

        return view('role-permission.role.add-permissions', [
            'role' => $role,
            'permissions' => $permissions,
            'rolePermissions' => $rolePermissions
        ]);
    }

    public function givePermissionToRole(Request $request, $roleId)
    {
        $request->validate([
            'permission' => 'required'
        ]);

        $role = Role::findOrFail($roleId);
        $role->syncPermissions($request->permission);

        return redirect()->back()->with('status','Permissions added to role');
    }
}