<?php

namespace App\Http\Controllers;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use App\Models\User;
use App\Models\Students;
use App\Models\BatchStudents;
use App\Models\Batches;
use App\Models\Lessons;
use App\Models\Tutors;
use App\Mail\StudentOnBoardParentMail;
use App\Mail\StudentOnBoardTutorMail;
use Illuminate\Support\Facades\DB;
use DataTables;
use Validator;
use Session;
use Auth;
use Hash;
use Mail;

class StudentController extends Controller{

    public function __construct(){
        $this->middleware('permission:view-student', ['only' => ['index','show']]);
        $this->middleware('permission:create-student', ['only' => ['create','store']]);
        $this->middleware('permission:update-student', ['only' => ['edit','store']]);
        $this->middleware('permission:delete-student', ['only' => ['destroy']]);
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request){
        try {
            $data = [];
            $data['page_title']        = 'Students List';
            $students = null;
            if($request->ajax()){
                if(Auth::user()->role_name == RoleTutor){
                    $user_id = Auth::user()->id;
                    $tutor = Tutors::where('user_id', $user_id)->first();
                    $result = DB::select("SELECT students_id FROM batch_students WHERE batches_id IN(SELECT id FROM batches WHERE tutor_id = ".$tutor->id.")");

                    $students = Students::whereIn('id', collect($result)->pluck('students_id')->all())->whereNull('deleted_at');
                }
                else{
                    $students = Students::whereNull('deleted_at');
                }
                return DataTables::eloquent($students)
                    ->addColumn('action', function($students) {
                        $action      = '';
                        if(Auth::user()->can('view-student')){
                            $action .= '<a href="'.route("students.view",$students->id).'" title="View Student"><i class="menu-icon tf-icons ti ti-eye"></i></a>&nbsp;';
                        }
                        if(Auth::user()->can('update-student')){
                            $action .= '<a href="'.route("students.edit",$students->id).'" title="Edit Student"><i class="menu-icon tf-icons ti ti-edit"></i></a>&nbsp;';
                        }
                        if(Auth::user()->can('delete-student')){
                            $action .= '<a href="javascript:void(0);" id="tutor_id_'.$students->id.'" title="Delete Student" data-id="'.$students->id.'" onclick="deleteRecord(this,'.$students->id.');"><i class="menu-icon tf-icons ti ti-trash"></i></a>&nbsp;';
                        }
                    
                        return $action;
                    })
                    ->addColumn('name', function($students) {
                        return $students->first_name.' '.$students->last_name;
                    })
                    ->filterColumn('name', function ($query, $keyword) {
                        $query->where('first_name', 'LIKE', "{$keyword}%")
                            ->orWhere('last_name', 'LIKE', "{$keyword}%");
                    })
                    ->addColumn('batches', function($students) {
                        $batches = $students->batches()->get()->pluck('name')->toArray();
                        return (count($batches) > 0 ) 
                        ? implode(', ', $students->batches()->get()->pluck('name')->toArray())
                        : 'Not enrolled to any batch.';
                    })
                    ->editColumn('created_at', function($users) {
                        return ($users->created_at!=null)?date(config('constant.date_format'), strtotime($users->created_at)):'-';
                    })
                    ->editColumn('updated_at', function($users) {
                        return ($users->updated_at!=null)?date(config('constant.date_format'), strtotime($users->updated_at)):'-';
                    })
                    ->filterColumn('created_at', function ($query, $keyword) {
                        $query->whereRaw("DATE_FORMAT(created_at,'%d-%m-%Y %h:%i %r') like ?", ["%$keyword%"]);
                    })
                    ->filterColumn('updated_at', function ($query, $keyword) {
                        $query->whereRaw("DATE_FORMAT(updated_at,'%d-%m-%Y %h:%i %r') like ?", ["%$keyword%"]);
                    })
                    ->rawColumns([])
                    ->make(true);
            }
            return view('students.index',$data);
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    public function exists(Request $request){
        if(isset($request->id) && $request->id!=''){
            $result = Student::where('id', '!=' , $request->id)->count();
        }else{
            $result = Student::where('id',$request->id)->count();
        }
        if($result>0){
            return response()->json(false);
        }else{
            return response()->json(true);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(){
        try {
            $data['page_title']         = 'Add Student';
            $data['batches']            = Batches::whereHas('lessons', function($q){
                    $q->where('status', LessonStatusPlanned);
                })->whereNull('deleted_at')->get();

            return view('students.add',$data);
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request){
        try {
            $rules          = [
                'first_name'                    => 'required|string|max:255',
                'last_name'                     => 'required|string|max:255',
                'parent_name'                   => 'required|string|max:255',
                'parent_mobile'                 => 'required|string|max:255',
                'parent_email'                  => 'required|string|max:255',
            ];

            $messages       = [
                'first_name.required'           => 'The first name field is required.',
                'last_name.required'            => 'The last name field is required.',
                'parent_name.required'          => 'The parent name field is required.',
                'parent_mobile.required'        => 'The parent mobile field is required.',
                'parent_email.required' 	    => 'The parent email field is required.',
                'batches.required' 	            => 'The batches field is required.',
            ];
            $validator      = Validator::make($request->all(), $rules,$messages);

            if ($validator->fails()) {
                if (isset($request->id) && !empty($request->id)) {
                    return redirect()->route('students.edit',$request->id)
                                ->withErrors($validator)
                                ->withInput();
                }else{
                    return redirect()->route('students.add')
                                ->withErrors($validator)
                                ->withInput();
                }
            }else{
                $name                       = $request->first_name." ".$request->last_name;
                $role = Role::where('name', 'student')->whereNull('deleted_at')->first();
                $parent_mobile = str_replace(['(', ')', '-'], ['', '', ''], $request->parent_mobile);

                if (isset($request->id) && !empty($request->id)) {
                    $id                            = $request->id;
                    $student                       = Students::where('id',$id)->first();
                    $studentBatches                = $student->batches()->get()->pluck('id')->toArray();
                    $student->first_name           = $request->first_name;
                    $student->last_name            = $request->last_name;
                    $student->parent_name          = $request->parent_name;
                    $student->parent_email         = $request->parent_email;
                    $student->parent_mobile        = $request->parent_mobile;
                    $student->gender               = $request->gender;
                    $batches                       = $request->batches;
                    $student->updated_by           = Auth::user()->id;
                    $student->updated_at           = date("Y-m-d H:i:s");
                    if($student->save()){
                        $student->batches()->detach($studentBatches);
                        $student->batches()->attach($batches);
                        Session::flash('alert-message','Student updated successfully.');
                        Session::flash('alert-class','success');
                        return redirect()->route('students');
                    }else{
                        Session::flash('alert-message','Student updated unsuccessfully.');
                        Session::flash('alert-class','error');
                        return redirect()->route('students.edit',$id);
                    }
                }else{
                    if (User::where('email', $request->parent_email)->exists()) {
                        Session::flash('alert-message','email-id already exists.');
                        Session::flash('alert-class','warning');
                        return redirect()->route('students.add');
                    }
                    else{
                        $user                       = new User();
                        $user->name                 = trim($name);
                        $user->password             = bcrypt($parent_mobile);
                        $user->email                = trim($request->parent_email);
                        $user->email_verified_at    = date("Y-m-d H:i:s");
                        $user->created_by           = Auth::user()->id;
                        $user->created_at           = date("Y-m-d H:i:s");
                        if($user->save()){
                            $user->assignRole($role);
                            $student                       = new Students();
                            $student->user_id              = $user->id;
                            $student->first_name           = $request->first_name;
                            $student->last_name            = $request->last_name;
                            $student->parent_name          = $request->parent_name;
                            $student->parent_email         = $request->parent_email;
                            $student->parent_mobile        = $parent_mobile;
                            $student->gender               = $request->gender;
                            $batches                       = $request->batches;
                            $student->created_by           = Auth::user()->id;
                            $student->created_at           = date("Y-m-d H:i:s");
                            if($student->save()){
                                if($batches){
                                    $student->batches()->attach($batches);
                                }
                                
                                if(getSetting('ALLOW_MAIL_FEATURE') == 1){
                                    //Mail to Parent
                                    Mail::to(trim($request->parent_email))->send(new StudentOnBoardParentMail($student));
        
                                    if(isset($batches) && count($batches) > 0){
                                        foreach($batches as $id){
                                            $batch = Batches::where('id', $id)->first();
            
                                            //Mail to Tutor
                                            Mail::to(trim($batch->tutor->user->email))->send(new StudentOnBoardTutorMail($student, $batch));
                                        }
                                    }
                                }
                                
                                Session::flash('alert-message','Student added successfully.');
                                Session::flash('alert-class','success');
                                return redirect()->route('students');
                            }else{
                                Session::flash('alert-message','Student added unsuccessfully.');
                                Session::flash('alert-class','error');
                                return redirect()->route('students.add');
                            }
                        }else{
                            Session::flash('alert-message','Student added unsuccessfully.');
                            Session::flash('alert-class','error');
                            return redirect()->route('students.add');
                        }
                    }
                }
            }
        }catch (\Exception $e) {
            Session::flash('alert-message',$e->getMessage());
            Session::flash('alert-class','error');
            return redirect()->route('students');
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id){
        try {
            $data['page_title']         = 'View Student';
            $student                      = Students::where('id',$id)->first();
            if($student){
                //$data['user']   = $user;
                $data['student']   = $student;
                return view('students.view',$data);
            }else{
                return abort(404);
            }
            
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id){
        try {
            $data['page_title']         = 'Edit Student';
            $data['batches']            = Batches::whereHas('lessons', function($q){
                    $q->where('status', LessonStatusPlanned);
                })->whereNull('deleted_at')->get();
            $student                    = Students::where('id', $id)->first();
            $data['studentBatches']     = $student->batches()->get()->pluck('id')->toArray();
            if($student){
                $data['student']           = $student;
                return view('students.edit',$data);
            }else{
                return abort(404);
            }
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request){
        if ($request->ajax()) {
            try {
                $student                                   = Students::where('id',$request->id)->first();
                if(!is_null($student)){
                    $student->disableLogging();
                    $student->deleted_by           = Auth::user()->id;
                    $student->save();
                    $student->enableLogging();
                    if($student->delete()){
                        if($student->user) $student->user->delete();
                        
                        $response['success']    = true;
                        $response['message']    = "Student deleted successfully.";
                    }else{
                        $response['success']    = false;
                        $response['message']    = "Student deleted unsuccessfully.";
                    }
                }else{
                    $response['success']                = false;
                    $response['message']                = "Student record not found.";
                }
            } catch (\Exception $e) {
                $response['success']                    = false;
                $response['message']                    = $e->getMessage();
            }
            return response()->json($response);
        }else{
            return abort(404);
        }
    }

    public function batches(Request $request){
        try {
            if($request->ajax()){ 
                $student = Student::where('id', $request->id)->first();
                $batches = Batches::whereHas('students', function($q) use($student){
                            $q->where('student_id', $student->id);
                        })->whereNull('deleted_at');
                return DataTables::eloquent($batches)
                    ->addColumn('action', function($batches) {
                        $action      = '';
                        if(Auth::user()->can('view-batch')){
                            $action .= '<a href="'.route("batches.view",$batches->id).'" title="View Batch"><i class="menu-icon tf-icons ti ti-eye"></i></a>&nbsp;';
                        }
                    
                        return $action;
                    })
                    ->addColumn('tutor_id', function($batches) {
                        return $batches->tutor->first_name." ".$batches->tutor->last_name;
                    })
                    ->filterColumn('tutor_id', function ($query, $keyword) {
                        $query->whereHas('tutor', function ($q) use ($keyword) {
                            $q->where('first_name', 'LIKE', "{$keyword}%")
                            ->where('last_name', 'LIKE', "{$keyword}%");
                        });
                    })
                    ->addColumn('lessons', function($batches) {
                        return $batches->lessons->count();
                    })
                    ->editColumn('created_at', function($users) {
                        return ($users->created_at!=null)?date(config('constant.date_format'), strtotime($users->created_at)):'-';
                    })
                    ->editColumn('updated_at', function($users) {
                        return ($users->updated_at!=null)?date(config('constant.date_format'), strtotime($users->updated_at)):'-';
                    })
                    ->filterColumn('created_at', function ($query, $keyword) {
                        $query->whereRaw("DATE_FORMAT(created_at,'%d-%m-%Y %h:%i %r') like ?", ["%$keyword%"]);
                    })
                    ->filterColumn('updated_at', function ($query, $keyword) {
                        $query->whereRaw("DATE_FORMAT(updated_at,'%d-%m-%Y %h:%i %r') like ?", ["%$keyword%"]);
                    })
                    ->rawColumns([])
                    ->make(true);
            }
            else{
            }
        }
        catch (\Exception $e) {
            return abort(404);
        }

    }
  
}
