<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use App\Models\Tickets;
use App\Models\TicketTimelines;
use App\Models\User;

use DataTables;
use Validator;
use Session;
use Auth;


class TicketController extends Controller{

    public function __construct(){
        $this->middleware('permission:view-ticket', ['only' => ['index','show']]);
        $this->middleware('permission:create-ticket', ['only' => ['create','store']]);
        $this->middleware('permission:update-ticket', ['only' => ['edit','update']]);
        $this->middleware('permission:delete-ticket', ['only' => ['destroy']]);

    }
    
    public function index(Request $request)
    {
        try {
            $data = [];
            $data['page_title'] = 'Tickets';
            $data['users'] = User::all();

            if($request->ajax()){
                $query = Tickets::query()->where('assigned_to', Auth::user()->id)->select('*');

                if($request->status != ''){
                    $query->where('status', $request->status);
                }
        
                if($request->priority != ''){
                    $query->where('priority', $request->priority);
                }
        
                if($request->assigned_to != ''){
                    $query->where('assigned_to', $request->assigned_to);
                }
                else{
                    $query->where('assigned_to', '=', Auth::user()->id)
                    ->orWhere('assigned_by', '=', Auth::user()->id);
                }
                
                return Datatables::eloquent($query)
                    ->addColumn('action', function ($query) {
                        $action      = '';
                        if(Auth::user()->can('view-ticket')){
                            $action .= '<a href="'.route("tickets.view",$query->id).'" title="View Ticket"><i class="menu-icon tf-icons ti ti-eye"></i></a>&nbsp;';
                        }
                        if(Auth::user()->can('update-ticket')){
                            $action .= '<a href="'.route("tickets.edit",$query->id).'" title="Edit Ticket"><i class="menu-icon tf-icons ti ti-edit"></i></a>&nbsp;';
                        }
                    
                        return $action;
                    })
                    ->editColumn('number', function ($query) {
                        $content = '';
                        if($query->priority == TicketPriorityLow){
                            $content = "<div class='badge bg-success rounded-pill ms-auto'>".$query->number."</div>";
                        }
                        if($query->priority == TicketPriorityMedium){
                            $content = "<div class='badge bg-warning rounded-pill ms-auto'>".$query->number."</div>";
                        }
                        if($query->priority == TicketPriorityHigh){
                            $content = "<div class='badge bg-danger rounded-pill ms-auto'>".$query->number."</div>";
                        }
                        return $content;
                    })
                    ->editColumn('priority', function ($query) {
                        $priority = '';
                        if($query->priority == TicketPriorityLow){
                            $priority = 'Low';
                        }
                        if($query->priority == TicketPriorityMedium){
                            $priority = 'Medium';
                        }
                        if($query->priority == TicketPriorityHigh){
                            $priority = 'High';
                        }
                        return $priority;
                    })
                    ->filterColumn('priority', function ($query, $keyword) {
                        $data = collect(array_filter([
                            ['id'=>0, 'name'=>'Low'],
                            ['id'=>1, 'name'=>'Medium'],
                            ['id'=>2, 'name'=>'High'],
                        ], function ($item) use($keyword) {
                            return str_contains(strtolower($item['name']), strtolower($keyword));
                        }))->map(function ($status) {
                            return $status['id'];
                        })->toArray();

                        $query->whereIn('priority', $data);
                    })
                    ->editColumn('status', function ($query) {
                        $status = '';
                        if($query->status == TicketStatusCreated){
                            $status = 'Created';
                        }
                        if($query->status == TicketStatusAssigned){
                            $status = 'Assigned';
                        }
                        if($query->status == TicketStatusClosed){
                            $status = 'Closed';
                        }
                        if($query->status == TicketStatusReopen){
                            $status = 'Reopen';
                        }
                        return $status;
                    })
                    ->filterColumn('status', function ($query, $keyword) {
                        $data = collect(array_filter([
                            ['id'=>0, 'name'=>'Created'],
                            ['id'=>1, 'name'=>'Assigned'],
                            ['id'=>2, 'name'=>'Closed'],
                            ['id'=>3, 'name'=>'Reopen'],
                        ], function ($item) use($keyword) {
                            return str_contains(strtolower($item['name']), strtolower($keyword));
                        }))->map(function ($status) {
                            return $status['id'];
                        })->toArray();

                        $query->whereIn('status', $data);
                    })
                    ->editColumn('created_at', function ($query) {
                        return $query->created_at != null ? date(config('constant.date_format_new'), strtotime($query->created_at)) : '-';
                    })
                    ->editColumn('created_by', function ($query) {
                        return $query->_created_by ? $query->_created_by->name : '-';
                    })
                    ->editColumn('assigned_at', function ($query) {
                        return $query->assigned_at != null ? date(config('constant.date_format_new'), strtotime($query->assigned_at)) : '-';
                    })
                    ->editColumn('assigned_to', function ($query) {
                        return $query->_assigned_to ? $query->_assigned_to->name : '-';
                    })
                    ->rawColumns(['action', 'number'])
                    ->make(true);
            
            }

            return view('tickets.index', $data);
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    public function create()
    {
        try {
            $data['page_title'] = 'Add';
            $data['ticket_number'] = generate_ticket_no();
            
            return view('tickets.add', $data);
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    public function store(Request $request)
    {
        try {
            $rules = [
                'title' => 'required',
                'priority' => 'required',
            ];
            $messages = [
                'title.required' => 'Title field is required.',
                'priority.required' => 'Priority field is required.',
            ];

            $validator = Validator::make($request->all(), $rules, $messages);
            if ($validator->fails()) {
                if ($request->ajax()) {
                    return response()->json(
                        [
                            'success' => false,
                            'message' => $validator->messages()->toArray(),
                        ],
                        200
                    );
                } else {
                    return redirect()
                        ->back()
                        ->withErrors($validator)
                        ->withInput();
                }
            } else {
                if ($request->hasFile("attachment")) {
                    $file = $request->file("attachment");
                    $attachment = time() . "." . $file->getClientOriginalExtension();
                    $destinationPath = public_path(config("constant.ticket_attach"));
                    $file->move($destinationPath, $attachment);
                }
                
                if (isset($request->id) && !empty($request->id)) {
                    $ticket = Tickets::where('id', $request->id)->first();

                    $ticket->title = $request->title;
                    $ticket->description = $request->description;
                    $ticket->priority = $request->priority;
                    $ticket->is_read = 0;
                    if (isset($attachment)) {
                        $ticket->attachment = $attachment;
                    }
                    $ticket->save();

                    Session::flash('alert-class', 'success');
                    Session::flash('alert-message', 'Updated successfully.');
                    return redirect()->route('tickets');
                } else {
                    //Add Ticket

                    $ticket = new Tickets();
                    $ticket->number = $request->number;
                    $ticket->title = $request->title;
                    $ticket->description = $request->description;
                    $ticket->priority = $request->priority;
                    $ticket->status = TicketStatusAssigned;
                    $ticket->is_read = 0;
                    if (isset($attachment)) {
                        $ticket->attachment = $attachment;
                    }

                    $ticket->created_at = date("Y-m-d H:i:s");
                    $ticket->created_by = Auth::user()->id;
                    $ticket->assigned_to = config("constant.admin_user_id");
                    $ticket->assigned_by = Auth::user()->id;
                    $ticket->assigned_at = date("Y-m-d H:i:s");
                    $ticket->save();

                    Session::flash('alert-message', 'Added successfully.');
                    Session::flash('alert-class', 'success');
                    return redirect()->route('tickets');
                }
            }
        } catch (\Exception $e) {
            if ($request->ajax()) {
                return response()->json(
                    [
                        'success' => false,
                        'message' => $e->getMessage(),
                    ],
                    200
                );
            }
            Session::flash('alert-message', $e->getMessage());
            Session::flash('alert-class', 'error');
            return \Redirect::back();
        }
    }

    public function show($id)
    {
        try {
            $data['page_title'] = 'View Ticket';
            $ticket = Tickets::where('id', $id)
                ->with(['comments'])
                ->whereNull('deleted_at')
                ->first();
            if ($ticket) {
                //$data['gutka']           = $gutka;
                $data['ticket'] = $ticket;
                $priority = '';
                if($ticket->priority == TicketPriorityLow){
                    $priority = 'Low';
                }
                else if($ticket->priority == TicketPriorityMedium){
                    $priority = 'Medium';
                }
                else if($ticket->priority == TicketPriorityHigh){
                    $priority = 'High';
                }
                $data['priority'] = $priority;

                
                $status = '';
                if($ticket->status == TicketStatusCreated){
                    $status = 'Created';
                }
                else if($ticket->status == TicketStatusAssigned){
                    $status = 'Assigned';
                }
                else if($ticket->status == TicketStatusClosed){
                    $status = 'Closed';
                }
                else if($ticket->status == TicketStatusReopen){
                    $status = 'Reopen';
                }
                $data['status'] = $status;
                $data['users'] = User::all();

                if($ticket->assigned_to == Auth::user()->id){
                    $ticket->is_read = 1;
                    $ticket->save();
                }

                return view('tickets.view', $data);
            } else {
                return abort(404);
            }
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    public function edit($id)
    {            
        try {
            $data['page_title'] = 'Edit Ticket';

            $ticket = Tickets::where('id', $id)
                ->whereNull('deleted_at')
                ->first();

            if ($ticket) {
                $data['ticket'] = $ticket;
                return view('tickets.edit', $data);
            } else {
                return abort(404);
            }
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    public function assign(Request $request)
    {
        try {
            $rules = [
                'assigned_to' => 'required'
            ];
            $messages = [
                'assigned_to.required' => 'Assignee field is required.'
            ];

            $validator = Validator::make($request->all(), $rules, $messages);
            if ($validator->fails()) {
                if ($request->ajax()) {
                    return response()->json(
                        [
                            'success' => false,
                            'message' => $validator->messages()->toArray(),
                        ],
                        200
                    );
                } else {
                    return redirect()
                        ->back()
                        ->withErrors($validator)
                        ->withInput();
                }
            } else {
                $ticket = Tickets::where('id', $request->ticket_id)
                            ->with(['comments'])
                            ->whereNull('deleted_at')
                            ->first();
                if($ticket){
                    $ticket->assigned_to = $request->assigned_to;
                    $ticket->assigned_by = Auth::user()->id;
                    $ticket->assigned_at = date("Y-m-d H:i:s");
                    $ticket->updated_by = Auth::user()->id;
                    $ticket->updated_at = date("Y-m-d H:i:s");
                    $ticket->status = TicketStatusAssigned;
                    $ticket->save();
    
                   
                    $_assigned_to_roles = [];
                    foreach($ticket->_assigned_to->roles as $role)
                        array_push($_assigned_to_roles, $role->display_name);

                    TicketTimelines::create([
                        'ticket_id' => $ticket->id,
                        'comments' =>  $ticket->_assigned_by->name." has assigned ticket ".$ticket->number." to ".$ticket->_assigned_to->name.'('.implode(',', $_assigned_to_roles).')',
                        'created_at' => date("Y-m-d H:i:s"),
                        'created_by' => Auth::user()->id
                    ]);
                   
                    Session::flash('alert-message', 'Ticket assigned successfully.');
                    Session::flash('alert-class', 'success');
                    return redirect()->route('tickets.view', $ticket->id);
                }
                else{
                    return redirect()->route('tickets');
                }
                
            }
        } catch (\Exception $e) {
            if ($request->ajax()) {
                return response()->json(
                    [
                        'success' => false,
                        'message' => $e->getMessage(),
                    ],
                    200
                );
            }
            Session::flash('alert-message', $e->getMessage());
            Session::flash('alert-class', 'error');
            return \Redirect::back();
        }
    }

    public function comment(Request $request)
    {
        try {
            $rules = [
                'comments' => 'required',
            ];
            $messages = [
                'comments.required' => 'Comment field is required.',
            ];

            $validator = Validator::make($request->all(), $rules, $messages);
            if ($validator->fails()) {
                if ($request->ajax()) {
                    return response()->json(
                        [
                            'success' => false,
                            'message' => $validator->messages()->toArray(),
                        ],
                        200
                    );
                } else {
                    return redirect()
                        ->back()
                        ->withErrors($validator)
                        ->withInput();
                }
            } else {
                
                TicketTimelines::create([
                    'ticket_id' => $request->ticket_id,
                    'comments' =>  $request->comments,
                    'created_at' => date("Y-m-d H:i:s"),
                    'created_by' => Auth::user()->id
                ]);


                Session::flash('alert-message', 'Comment added successfully.');
                Session::flash('alert-class', 'success');
                return redirect()->route('tickets.view',  $request->ticket_id);
            }
        } catch (\Exception $e) {
            if ($request->ajax()) {
                return response()->json(
                    [
                        'success' => false,
                        'message' => $e->getMessage(),
                    ],
                    200
                );
            }
            Session::flash('alert-message', $e->getMessage());
            Session::flash('alert-class', 'error');
            return \Redirect::back();
        }
    }

    public function statusUpdate(Request $request)
    {
        try {
            $rules = [
                'status' => 'required'
            ];
            $messages = [
                'status.required' => 'Status field is required.'
            ];

            $validator = Validator::make($request->all(), $rules, $messages);
            if ($validator->fails()) {
                if ($request->ajax()) {
                    return response()->json(
                        [
                            'success' => false,
                            'message' => $validator->messages()->toArray(),
                        ],
                        200
                    );
                } else {
                    return redirect()
                        ->back()
                        ->withErrors($validator)
                        ->withInput();
                }
            } else {
                $ticket = Tickets::where('id', $request->ticket_id)
                            ->with(['comments'])
                            ->whereNull('deleted_at')
                            ->first();
                if($ticket){
                    $status = '';
                    if($request->status == TicketStatusCreated){
                        $status = 'Created';
                    }
                    if($request->status == TicketStatusAssigned){
                        $status = 'Assigned';
                    }
                    if($request->status == TicketStatusClosed){
                        $status = 'Closed';
                    }
                    if($request->status == TicketStatusReopen){
                        $status = 'Reopen';
                    }

                    $ticket->status = $request->status;
                    $ticket->updated_by = Auth::user()->id;
                    $ticket->updated_at = date("Y-m-d H:i:s");
                    $ticket->save();
                   
                    TicketTimelines::create([
                        'ticket_id' => $ticket->id,
                        'comments' =>  $ticket->_assigned_by->name." has update the status of ticket to ".$status,
                        'created_at' => date("Y-m-d H:i:s"),
                        'created_by' => Auth::user()->id
                    ]);
                   
                    Session::flash('alert-message', 'Ticket updated successfully.');
                    Session::flash('alert-class', 'success');
                    return redirect()->route('tickets.view', $ticket->id);
                }
                else{
                    return redirect()->route('tickets');
                }
                
            }
        } catch (\Exception $e) {
            if ($request->ajax()) {
                return response()->json(
                    [
                        'success' => false,
                        'message' => $e->getMessage(),
                    ],
                    200
                );
            }
            Session::flash('alert-message', $e->getMessage());
            Session::flash('alert-class', 'error');
            return \Redirect::back();
        }
    }
}
