<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use DataTables;

class UserController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:view-user', ['only' => ['index']]);
        $this->middleware('permission:create-user', ['only' => ['create','store']]);
        $this->middleware('permission:update-user', ['only' => ['update','edit']]);
        $this->middleware('permission:delete-user', ['only' => ['destroy']]);
    }
/**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request){
        try {
            $data = [];
            $data['page_title']        = 'User List';
            if($request->ajax()){
                $users = User::query();
                return DataTables::eloquent($users)
                    ->addColumn('action', function($users) {
                        $action      = '';
                        if(Auth::user()->can('view-user')){
                            $action .= '<a href="'.route("users.view",$users->id).'" title="View User"><i class="menu-icon tf-icons ti ti-eye"></i></a>&nbsp;';
                        }
                        return $action;
                    })
                    ->addColumn('role', function($users) {
                        return $users->role_name;
                    })
                    ->editColumn('created_at', function($users) {
                        return ($users->created_at!=null)?date(config('constant.date_format'), strtotime($users->created_at)):'-';
                    })
                    ->editColumn('updated_at', function($users) {
                        return ($users->updated_at!=null)?date(config('constant.date_format'), strtotime($users->updated_at)):'-';
                    })
                    ->filterColumn('created_at', function ($query, $keyword) {
                        $query->whereRaw("DATE_FORMAT(created_at,'%d-%m-%Y %h:%i %r') like ?", ["%$keyword%"]);
                    })
                    ->filterColumn('updated_at', function ($query, $keyword) {
                        $query->whereRaw("DATE_FORMAT(updated_at,'%d-%m-%Y %h:%i %r') like ?", ["%$keyword%"]);
                    })
                    ->rawColumns([])
                    ->make(true);
            }
            return view('users.index',$data);
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    public function exists(Request $request){
        if(isset($request->id) && $request->id!=''){
            $result = User::where('id', '!=' , $request->id)->where('email',$request->email)->count();
        }else{
            $result = User::where('email',$request->email)->count();
        }
        if($result>0){
            return response()->json(false);
        }else{
            return response()->json(true);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(){
        try {
            $data['page_title']         = 'Add User';
            $data['roles']              = Role::where('name','!=','super-admin')->whereNull('deleted_at')->get();
            return view('users.add',$data);
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request){
        try {
            $rules          = [
                'role_id'               => 'required',
                'name'                  => 'required|string|max:255'
            ];
            if (isset($request->id) && !empty($request->id)) {
                $rules['email']         = 'required|string|email|max:255|unique:users,email,'.$request->id;
            }else{
                $rules['email']         = 'required|string|email|max:255|unique:users';
            }
            $messages       = [
                'role_id.required'      => 'The role field is required.',
                'name.required'         => 'The fullname field is required.',
                'email.required'        => 'The email field is required.'
            ];
            $validator      = Validator::make($request->all(), $rules,$messages);
            if ($validator->fails()) {
                if (isset($request->id) && !empty($request->id)) {
                    return redirect()->route('users.edit',$request->id)
                                ->withErrors($validator)
                                ->withInput();
                }else{
                    return redirect()->route('users.add')
                                ->withErrors($validator)
                                ->withInput();
                }
            }else{
                $role                       = Role::where('id',$request->role_id)->whereNull('deleted_at')->first();
                $site_role = [];
                if($request->has('sites') && $request->has('roles')){
                    $sites = $request->sites;
                    $roles = $request->roles;
                    for ($i = 0; $i < count($sites); $i++) {
                        if(isset($sites[$i]) && $sites[$i] != "" && isset($roles[$i]) && $roles[$i] != "")
                            $site_role[] = ['site_id'=>$sites[$i],'role_id'=>$roles[$i],'created_at'=>date("Y-m-d H:i:s")];
                    }
                }
                if (isset($request->id) && !empty($request->id)) {
                    $id                     = $request->id;
                    $user                   = User::where('id',$id)->first();

                    if($request->password!=''){
                        $user->password     = bcrypt($request->password);
                    }
                    $user->name             = trim($request->name);
                    $user->email            = trim($request->email);
                    $user->status           = $request->status;
                    $user->updated_by       = Auth::user()->id;
                    $user->updated_at       = date("Y-m-d H:i:s");
                    $user->site_role()->sync($site_role);
                    /*$user->site_role()->detach();
                    $user->site_role()->attach($site_role);*/
                    if($user->save()){
                        $user->syncRoles($role->name);
                        Session::flash('alert-message','User updated successfully.');
                        Session::flash('alert-class','success');
                        return redirect()->route('users');
                    }else{
                        Session::flash('alert-message','User updated unsuccessfully.');
                        Session::flash('alert-class','error');
                        return redirect()->route('users.edit',$id);
                    }
                }else{
                    $user                   = new User();
                    $user->password         = bcrypt($request->password);
                    $user->name             = trim($request->name);
                    $user->email            = trim($request->email);
                    $user->email_verified_at= date("Y-m-d H:i:s");
                    $user->status           = $request->status;
                    $user->created_by       = Auth::user()->id;
                    $user->created_at       = date("Y-m-d H:i:s");
                    if($user->save()){

                        $user->site_role()->attach($site_role);

                        $user->assignRole($role->name);
                        Session::flash('alert-message','User added successfully.');
                        Session::flash('alert-class','success');
                        return redirect()->route('users');
                    }else{
                        Session::flash('alert-message','User added unsuccessfully.');
                        Session::flash('alert-class','error');
                        return redirect()->route('users.add');
                    }
                }
            }
        }catch (\Exception $e) {
            Session::flash('alert-message',$e->getMessage());
            Session::flash('alert-class','error');
            return redirect()->route('users');
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id){
        try {
            $data['page_title']         = 'View User';
            $user               = User::where('id',$id)->first();
            if($user){
                $data['user']   = $user;
                return view('users.view',$data);
            }else{
                return abort(404);
            }
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id){
        try {
            $data['page_title']         = 'Edit User';
            $user                       = User::where('id',$id)->first();
            if($user){
                if(!(in_array('super-admin', $user->getRoleNames()->toArray()))){
                    $data['roles']      = Role::where([
                        ['name', '!=', 'super-admin']
                    ])->whereNull('deleted_at')->get();
                }else{
                    $data['roles']      = Role::where('name','super-admin')->whereNull('deleted_at')->limit(1)->get();
                }
                $data['user']           = $user;
                $data['sites']   = Sitemanagements::query()->whereNull('deleted_at')->get();
                $data['site_role'] = $user->site_role;
                return view('users.add',$data);
            }else{
                return abort(404);
            }
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request){
        if ($request->ajax()) {
            try {
                $user                                   = User::where('id',$request->id)->first();
                if(!is_null($user)){
                    if(!(in_array('super-admin', $user->getRoleNames()->toArray()))){
                        if(Auth::user()->id!=$user->id){
                            $user->disableLogging();
                            $user->deleted_by           = Auth::user()->id;
                            $user->save();
                            $user->enableLogging();
                            if($user->delete()){
                                $response['success']    = true;
                                $response['message']    = "User deleted successfully.";
                            }else{
                                $response['success']    = false;
                                $response['message']    = "User deleted unsuccessfully.";
                            }
                        }else{
                            $response['success']        = false;
                            $response['message']        = "Can't delete user because ".$user->name." is active.";
                        }
                    }else{
                        $response['success']            = false;
                        $response['message']            = "Can't delete user because ".$user->name." is super ";
                    }
                }else{
                    $response['success']                = false;
                    $response['message']                = "User record not found.";
                }
            } catch (\Exception $e) {
                $response['success']                    = false;
                $response['message']                    = $e->getMessage();
            }
            return response()->json($response);
        }else{
            return abort(404);
        }
    }

    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email',
            'password' => 'required',
        ]);

        if($validator->fails())
        {
            return response()->json([
                'status' => false,
                'messages' => $validator->messages()->toArray()
            ], 200);
        }

        $credentials = $request->only('email', 'password');
        
        if (Auth::attempt($credentials)) {
            request()->session()->regerate();
            return redirect()->intended('/');
        }
    
        return redirect('/login')->with('error', 'Invalid credentials. Please try again.');
    }

    public function logout()
    {
        Auth::logout();
        request()->session()->invalidate();
    
        return redirect('/login');
    }

    public function forgotPassword(Request $request){
        try {
            $data['page_title']         = 'Forgot Password';
            return view('auth.forgot-password',$data);
        } catch (\Exception $e) {
            return abort(404);
        }
    }

    public function resetPassword(Request $request){
        try {
            $data['page_title']         = 'Forgot Password';
            return view('auth.forgot-password',$data);
        } catch (\Exception $e) {
            return abort(404);
        }
    }
}