import type { EnhancedEventEmitter } from './enhancedEvents';
import type { WebRtcServer, WebRtcServerOptions } from './WebRtcServerTypes';
import type { Router, RouterOptions } from './RouterTypes';
import type { AppData } from './types';
export type WorkerLogLevel = 'debug' | 'warn' | 'error' | 'none';
export type WorkerLogTag = 'info' | 'ice' | 'dtls' | 'rtp' | 'srtp' | 'rtcp' | 'rtx' | 'bwe' | 'score' | 'simulcast' | 'svc' | 'sctp' | 'message';
export type WorkerSettings<WorkerAppData extends AppData = AppData> = {
    /**
     * Logging level for logs generated by the media worker subprocesses (check
     * the Debugging documentation). Valid values are 'debug', 'warn', 'error' and
     * 'none'. Default 'error'.
     */
    logLevel?: WorkerLogLevel;
    /**
     * Log tags for debugging. Check the meaning of each available tag in the
     * Debugging documentation.
     */
    logTags?: WorkerLogTag[];
    /**
     * Minimun RTC port for ICE, DTLS, RTP, etc. Default 10000.
     * @deprecated Use |portRange| in TransportListenInfo object instead.
     */
    rtcMinPort?: number;
    /**
     * Maximum RTC port for ICE, DTLS, RTP, etc. Default 59999.
     * @deprecated Use |portRange| in TransportListenInfo object instead.
     */
    rtcMaxPort?: number;
    /**
     * Path to the DTLS public certificate file in PEM format. If unset, a
     * certificate is dynamically created.
     */
    dtlsCertificateFile?: string;
    /**
     * Path to the DTLS certificate private key file in PEM format. If unset, a
     * certificate is dynamically created.
     */
    dtlsPrivateKeyFile?: string;
    /**
     * Field trials for libwebrtc.
     * @private
     *
     * NOTE: For advanced users only. An invalid value will make the worker crash.
     * Default value is
     * "WebRTC-Bwe-AlrLimitedBackoff/Enabled/".
     */
    libwebrtcFieldTrials?: string;
    /**
     * Disable liburing (io_uring) despite it's supported in current host.
     */
    disableLiburing?: boolean;
    /**
     * Custom application data.
     */
    appData?: WorkerAppData;
};
export type WorkerUpdateableSettings<T extends AppData = AppData> = Pick<WorkerSettings<T>, 'logLevel' | 'logTags'>;
/**
 * An object with the fields of the uv_rusage_t struct.
 *
 * - http://docs.libuv.org/en/v1.x/misc.html#c.uv_rusage_t
 * - https://linux.die.net/man/2/getrusage
 */
export type WorkerResourceUsage = {
    /**
     * User CPU time used (in ms).
     */
    ru_utime: number;
    /**
     * System CPU time used (in ms).
     */
    ru_stime: number;
    /**
     * Maximum resident set size.
     */
    ru_maxrss: number;
    /**
     * Integral shared memory size.
     */
    ru_ixrss: number;
    /**
     * Integral unshared data size.
     */
    ru_idrss: number;
    /**
     * Integral unshared stack size.
     */
    ru_isrss: number;
    /**
     * Page reclaims (soft page faults).
     */
    ru_minflt: number;
    /**
     * Page faults (hard page faults).
     */
    ru_majflt: number;
    /**
     * Swaps.
     */
    ru_nswap: number;
    /**
     * Block input operations.
     */
    ru_inblock: number;
    /**
     * Block output operations.
     */
    ru_oublock: number;
    /**
     * IPC messages sent.
     */
    ru_msgsnd: number;
    /**
     * IPC messages received.
     */
    ru_msgrcv: number;
    /**
     * Signals received.
     */
    ru_nsignals: number;
    /**
     * Voluntary context switches.
     */
    ru_nvcsw: number;
    /**
     * Involuntary context switches.
     */
    ru_nivcsw: number;
};
export type WorkerDump = {
    pid: number;
    webRtcServerIds: string[];
    routerIds: string[];
    channelMessageHandlers: {
        channelRequestHandlers: string[];
        channelNotificationHandlers: string[];
    };
    liburing?: {
        sqeProcessCount: number;
        sqeMissCount: number;
        userDataMissCount: number;
    };
};
export type WorkerEvents = {
    died: [Error];
    subprocessclose: [];
    listenererror: [string, Error];
    '@success': [];
    '@failure': [Error];
};
export type WorkerObserver = EnhancedEventEmitter<WorkerObserverEvents>;
export type WorkerObserverEvents = {
    close: [];
    newwebrtcserver: [WebRtcServer];
    newrouter: [Router];
};
export interface Worker<WorkerAppData extends AppData = AppData> extends EnhancedEventEmitter<WorkerEvents> {
    /**
     * Worker process identifier (PID).
     */
    get pid(): number;
    /**
     * Whether the Worker is closed.
     */
    get closed(): boolean;
    /**
     * Whether the Worker died.
     */
    get died(): boolean;
    /**
     * Whether the Worker subprocess is closed.
     */
    get subprocessClosed(): boolean;
    /**
     * App custom data.
     */
    get appData(): WorkerAppData;
    /**
     * App custom data setter.
     */
    set appData(appData: WorkerAppData);
    /**
     * Observer.
     */
    get observer(): WorkerObserver;
    /**
     * Close the Worker.
     */
    close(): void;
    /**
     * Dump Worker.
     */
    dump(): Promise<WorkerDump>;
    /**
     * Get mediasoup-worker process resource usage.
     */
    getResourceUsage(): Promise<WorkerResourceUsage>;
    /**
     * Update settings.
     */
    updateSettings(options?: WorkerUpdateableSettings<WorkerAppData>): Promise<void>;
    /**
     * Create a WebRtcServer.
     */
    createWebRtcServer<WebRtcServerAppData extends AppData = AppData>(options: WebRtcServerOptions<WebRtcServerAppData>): Promise<WebRtcServer<WebRtcServerAppData>>;
    /**
     * Create a Router.
     */
    createRouter<RouterAppData extends AppData = AppData>(options?: RouterOptions<RouterAppData>): Promise<Router<RouterAppData>>;
}
//# sourceMappingURL=WorkerTypes.d.ts.map