const os = require('os');
const fs = require('fs');
const path = require("path");
const mediasoup = require("mediasoup");
const siofu = require("socketio-file-upload");
const fetch = require("node-fetch");
const cron = require('node-cron');
let nextWorker = 0;
let meetings = {};
let sockets = {};
let transports = [];
let producers = [];
let consumers = [];
const workers = [];
const numberOfWorkers = Object.keys(os.cpus()).length;
const mediaCodecs = [{
  'kind': "audio",
  'mimeType': "audio/opus",
  'clockRate': 48000,
  'channels': 2
}, {
  'kind': 'video',
  'mimeType': "video/VP8",
  'clockRate': 90000,
  'parameters': {
    'x-google-start-bitrate': 1000
  }
}];
(async function runMediasoupWorkers() {
  for (let i = 1; i <= numberOfWorkers; i++) {
    const worker = await mediasoup.createWorker({
      'rtcMinPort': process.env.RTC_MIN_PORT,
      'rtcMaxPort': process.env.RTC_MAX_PORT
    });
    workers.push(worker);
  }
})();
function getWorker() {
  const worker = workers[nextWorker];
  if (++nextWorker === workers.length) {
    nextWorker = 0;
  }
  return worker;
}
async function getOrCreateMeeting(index) {
  let router;
  let users = [];
  let meeting = meetings[index];
  if (meeting && meeting.router) {
    router = meetings[index].router;
  } else {
    const worker = getWorker();
    router = await worker.createRouter({
      'mediaCodecs': mediaCodecs
    });
    meetings[index] = {
      'router': router,
      'users': users,
      ...meetings[index]
    };
  }
  return router;
}
async function handleJoin(server, socket, data) {
  let meetingId = data.meetingId;
  const router = await getOrCreateMeeting(meetingId);
  socket.join(meetingId);
  socket.meetingId = meetingId;
  socket.isTutor = data.isTutor;
  socket.userId = data.userId;
  socket.username = data.username;
  handleFileTransfer(socket, meetingId);
  sockets[socket.id] = {
    'socket': socket,
    'consumers': [],
    'producers': [],
    'transports': []
  };
  sendToPeer(server, {
    'type': 'usernames',
    'toSocketId': socket.id,
    'usernames': meetings[meetingId].users
  });
  meetings[meetingId].users.push(data.username);
  sendToMeeting(socket, {
    'type': "userJoined",
    'username': data.username,
    'id': socket.id
  });
  return router.rtpCapabilities;
}
function handleCheckMeeting(socket, data, server) {
  let response;
  let meetingRoom = !server.sockets.adapter.rooms.get(data.meetingId) || server.sockets.adapter.rooms.get(data.meetingId).size < data.userLimit;
  if (meetingRoom) {
    if (data.tutor) {
      meetings[data.meetingId] = {
        'isTutorPresent': true,
        'tutor': socket.id
      };
      response = {
        'result': true,
        'message': ''
      };
    } else {
      if (data.authMode == "disabled" || data.moderatorRights == 'disabled') {
        response = {
          'result': true,
          'message': ''
        };
      } else if (meetings[data.meetingId] && meetings[data.meetingId].isTutorPresent) {
        sendToPeer(server, {
          'type': 'permission',
          'toSocketId': meetings[data.meetingId].tutor,
          'fromSocketId': socket.id,
          'username': data.username
        });
        response = {
          'type': "info",
          'message': "please_wait"
        };
      } else {
        response = {
          'result': false,
          'message': "not_started"
        };
      }
    }
  } else {
    response = {
      'result': false,
      'message': "meeting_full"
    };
  }
  return response;
}
async function createWebRtcTransport(router) {
  return new Promise(async (successCallback, errorCallback) => {
    try {
      let webRtcTransport = await router.createWebRtcTransport({
        'listenIps': [{
          'ip': process.env.IP,
          'announcedIp': process.env.ANNOUNCED_IP
        }]
      });
      webRtcTransport.on("dtlsstatechange", dtlsState => {
        if (dtlsState === "closed") {
          webRtcTransport.close();
        }
      });
      webRtcTransport.on("close", () => {});
      successCallback(webRtcTransport);
    } catch (error) {
      errorCallback(error);
    }
  });
}
function addTransport(transport, meetingId, consumer, socketId) {
  transports = [...transports, {
    'socketId': socketId,
    'transport': transport,
    'meetingId': meetingId,
    'consumer': consumer
  }];
  sockets[socketId] = {
    ...sockets[socketId],
    'transports': [...sockets[socketId].transports, transport.id]
  };
}
function getTransport(socketId) {
  const [result] = transports.filter(transport => transport.socketId === socketId && !transport.consumer);
  return result.transport;
}
async function handleTransportProduce(data, socketId, meetingId) {
  const producer = await getTransport(socketId).produce({
    'kind': data.kind,
    'rtpParameters': data.rtpParameters,
    'appData': data.appData
  });
  addProducer(producer, meetingId, socketId);
  sendToMeeting(sockets[socketId].socket, {
    'type': "newProducer",
    'producerId': producer.id
  });
  producer.on("transportclose", () => {
    producer.close();
  });
  return producer.id;
}
function handleGetProducers(meetingId, socketId) {
  let items = [];
  producers.forEach(p => {
    if (p.socketId !== socketId && p.meetingId === meetingId) {
      items = [...items, p.producer.id];
    }
  });
  return items;
}
function addProducer(producer, meetingId, socketId) {
  producers = [...producers, {
    'socketId': socketId,
    'producer': producer,
    'meetingId': meetingId
  }];
  sockets[socketId] = {
    ...sockets[socketId],
    'producers': [...sockets[socketId].producers, producer.id]
  };
}
function addConsumer(consumer, meetingId, socketId) {
  consumers = [...consumers, {
    'socketId': socketId,
    'consumer': consumer,
    'meetingId': meetingId
  }];
  sockets[socketId] = {
    ...sockets[socketId],
    'consumers': [...sockets[socketId].consumers, consumer.id]
  };
}
async function handleConsume(data, socketId, meetingId) {
  try {
    const router = meetings[meetingId].router;
    let transport = transports.find(t => t.consumer && t.transport.id == data.serverConsumerTransportId).transport;
    if (router.canConsume({
      'producerId': data.remoteProducerId,
      'rtpCapabilities': data.rtpCapabilities
    })) {
      const consumer = await transport.consume({
        'producerId': data.remoteProducerId,
        'rtpCapabilities': data.rtpCapabilities,
        'paused': true
      });
      const producerSocketId = getProducerSocketId(data, meetingId);
      let result = producers.find(p => p.producer.id == data.remoteProducerId);
      consumer.on("transportclose", () => {});
      consumer.on("producerclose", () => {
        sockets[socketId].socket.emit('message', {
          'type': "producerClosed",
          'remoteProducerId': data.remoteProducerId,
          'producerSocketId': producerSocketId,
          'trackType': result.producer.appData
        });
        transport.close([]);
        transports = transports.filter(t => t.transport.id !== transport.id);
        consumer.close();
        consumers = consumers.filter(c => c.consumer.id !== consumer.id);
      });
      addConsumer(consumer, meetingId, socketId);
      return {
        'id': consumer.id,
        'producerId': data.remoteProducerId,
        'kind': consumer.kind,
        'rtpParameters': consumer.rtpParameters,
        'serverConsumerId': consumer.id,
        'producerSocketId': producerSocketId,
        'appData': result.producer.appData
      };
    }
  } catch (error) {
    return {
      'params': {
        'e': error
      }
    };
  }
}
function getProducerSocketId(data, meetingId) {
  return producers.find(p => p.producer.id === data.remoteProducerId && p.meetingId === meetingId).socketId;
}
async function handleConsumerResume(consumerId) {
  const {
    'consumer': consumer
  } = consumers.find(c => c.consumer.id === consumerId);
  await consumer.resume();
}
async function handleTransportRecvConnect(data) {
  const transport = transports.find(t => t.consumer && t.transport.id == data.serverConsumerTransportId).transport;
  await transport.connect({
    'dtlsParameters': data.dtlsParameters
  });
}
function removeItems(collection, socketId, type) {
  collection.forEach(item => {
    if (item.socketId === socketId) {
      item[type].close();
    }
  });
  collection = collection.filter(c => c.socketId !== socketId);
  return collection;
}
function handleProducerClose(socketId, producerId) {
  producers.forEach(p => {
    if (p.socketId === socketId && p.producer.id === producerId) {
      p.producer.close();
    }
  });
  producers = producers.filter(p => !(p.socketId == socketId && p.producer.id == producerId));
  sockets[socketId].producers = sockets[socketId].producers.filter(p => p !== producerId);
}
function sendToMeeting(socket, data) {
  socket.broadcast.to(socket.meetingId).emit('message', data);
}
function handleFileTransfer(socket, meetingId) {
  let filepath = path.join(__dirname, "../public/file_uploads/");
  if (!fs.existsSync(filepath)) {
    fs.mkdirSync(filepath);
  }
  var uploader = new siofu();
  uploader.dir = path.join(__dirname, "../public/file_uploads/" + meetingId);
  if (!fs.existsSync(uploader.dir)) {
    fs.mkdirSync(uploader.dir);
  }
  uploader.maxFileSize = process.env.MAX_FILESIZE * 1024 * 1024;
  uploader.listen(socket);
  uploader.on("saved", function (event) {
    event.file.clientDetail.file = event.file.base;
    event.file.clientDetail.extension = event.file.meta.extension;
    event.file.clientDetail.username = event.file.meta.username;
    sendToMeeting(socket, {
      'type': "file",
      'file': event.file.base,
      'extension': event.file.meta.extension,
      'username': event.file.meta.username
    });
  });
  uploader.on("error", function (error) {});
}
function handleRecordingPermission(socket, data, server) {
  sendToPeer(server, {
    'type': "recordingPermission",
    'toSocketId': meetings[data.meetingId].tutor,
    'fromSocketId': socket.id,
    'username': data.username
  });
}
function handleFrame(socket, data) {
  sendToMeeting(socket, {
    'type': "handleFrame",
    'socketId': socket.id,
    'action': data.action,
    'id': data.id
  });
}
function handleCharacter(socket, data) {
  sendToMeeting(socket, {
    'type': "handleCharacter",
    'socketId': socket.id,
    'action': data.action,
    'id': data.id
  });
}
function sendToPeer(socket, message) {
  socket.to(message.toSocketId).emit("message", message);
}
function checkDetails() {
  fetch(process.env.DOMAIN + "/check-details").then(res => res.text()).then(res => {
    console.log(res);
    if (!res) {
      process.exit(1);
    }
  });
}
module.exports = function (server) {
  //checkDetails();
  //cron.schedule("0 0 * * 0", () => {
  //  checkDetails();
  //});
  server.sockets.on("connection", socket => {
    socket.on("message", async (data, callback) => {
      switch (data.type) {
        case "join":
          const rtpCapabilities = await handleJoin(server, socket, data);
          callback({
            'rtpCapabilities': rtpCapabilities
          });
          break;
        case "createWebRtcTransport":
          createWebRtcTransport(meetings[socket.meetingId].router).then(result => {
            callback({
              'params': {
                'id': result.id,
                'iceParameters': result.iceParameters,
                'iceCandidates': result.iceCandidates,
                'dtlsParameters': result.dtlsParameters,
                'producersExist': !!producers.length
              }
            });
            addTransport(result, socket.meetingId, data.consumer, socket.id);
          });
          break;
        case "transportConnect":
          getTransport(socket.id).connect({
            'dtlsParameters': data.dtlsParameters
          });
          break;
        case "transportProduce":
          const producerId = await handleTransportProduce(data, socket.id, socket.meetingId);
          callback({
            'id': producerId,
            'socketId': socket.id
          });
          break;
        case "getProducers":
          callback(handleGetProducers(socket.meetingId, socket.id));
          break;
        case "transportRecvConnect":
          handleTransportRecvConnect(data);
          break;
        case 'consume':
          const result = await handleConsume(data, socket.id, socket.meetingId);
          callback({
            'params': result
          });
          break;
        case "consumerResume":
          handleConsumerResume(data.serverConsumerId);
          break;
        case "producerClose":
          handleProducerClose(socket.id, data.id);
          break;
        case 'meetingMessage':
        case "raiseHand":
        case "clearWhiteboard":
        case "closeWhiteboard":
        case "whiteboard":
        case "openWhiteboard":
        case 'sync':
        case "recordingStarted":
        case "showMotivator":
        case "muteAll":
          sendToMeeting(socket, data);
          break;
        case "muteUser":
        case "updateMicObjectId":
          sendToMeeting(socket, {
            'toSocketId': socket.id,
            ...data
          });
          break;
        case "handleFrame":
          handleFrame(socket, data);
          break;
        case "handleCharacter":
          handleCharacter(socket, data);
          break;
        case "recordingPermission":
          handleRecordingPermission(socket, data, server);
          break;
        case "recordingPermissionResult":
        case 'permissionResult':
        case "kick":
          sendToPeer(server, data);
          break;
        case "checkMeeting":
          callback(handleCheckMeeting(socket, data, server));
          break;
      }
    });
    socket.on('disconnect', () => {
      const meetingId = socket.meetingId;
      delete sockets[socket.id];
      let filepath = path.join(__dirname, "../public/file_uploads/" + meetingId);
      if (!server.sockets.adapter.rooms.get(meetingId) && fs.existsSync(filepath)) {
        fs.rmdirSync(filepath, {
          'recursive': true
        });
      }
      if (meetings[meetingId]) {
        const index = meetings[meetingId].users.indexOf(socket.username);
        if (index > -1) {
          meetings[meetingId].users.splice(index, 1);
        }
      }
      socket.leave(meetingId);
      sendToMeeting(socket, {
        'type': "leave",
        'socketId': socket.id,
        'meetingId': socket.meetingId,
        'isTutor': socket.isTutor,
        'userId': socket.userId,
        'username': socket.username
      });
      consumers = removeItems(consumers, socket.id, 'consumer');
      producers = removeItems(producers, socket.id, "producer");
      transports = removeItems(transports, socket.id, 'transport');
      if (socket.isTutor) {
        delete meetings[meetingId];
      }
    });
  });
};