<?php
namespace App\Http\Controllers;
use App\Models\Task;
use App\Models\TaskTimeEntry;
use Illuminate\Http\Request;
use Carbon\Carbon;

class TaskController extends Controller
{
    public function index(Request $request)
    {
        $user = $request->user();
        $tasks = Task::with('assignee')->orderBy('created_at','desc')->get();
        return view('tasks', compact('tasks'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'title'=>'required|string',
            'description'=>'nullable|string',
            'assigned_to'=>'nullable|exists:users,id'
        ]);
        $task = Task::create($data);
        return redirect()->back()->with('success','Task created');
    }

    public function start(Request $request, $id)
    {
        $task = Task::findOrFail($id);
        $user = $request->user();
        $entry = TaskTimeEntry::create([
            'task_id'=>$task->id,
            'user_id'=>$user->id,
            'date'=>Carbon::now()->toDateString(),
            'start_time'=>Carbon::now()
        ]);
        $task->status = 'in-progress';
        $task->save();
        return redirect()->back()->with('success','Timer started');
    }

    public function stop(Request $request, $id)
    {
        $entry = TaskTimeEntry::where('task_id',$id)->whereNull('end_time')->where('user_id',$request->user()->id)->latest()->first();
        if(!$entry) return redirect()->back()->with('error','No running timer found');
        $entry->end_time = Carbon::now();
        $entry->duration_minutes = Carbon::parse($entry->start_time)->diffInMinutes($entry->end_time);
        $entry->save();
        return redirect()->back()->with('success','Timer stopped');
    }

    public function manualEntry(Request $request)
    {
        $data = $request->validate([
            'task_id'=>'required|exists:tasks,id',
            'user_id'=>'required|exists:users,id',
            'start_time'=>'required|date',
            'end_time'=>'nullable|date',
            'note'=>'nullable|string'
        ]);
        $entry = TaskTimeEntry::create([
            'task_id'=>$data['task_id'],
            'user_id'=>$data['user_id'],
            'date'=>\Carbon\Carbon::parse($data['start_time'])->toDateString(),
            'start_time'=>$data['start_time'],
            'end_time'=>$data['end_time'] ?? null,
            'duration_minutes'=> isset($data['end_time']) ? \Carbon\Carbon::parse($data['start_time'])->diffInMinutes(\Carbon\Carbon::parse($data['end_time'])) : null,
            'note'=>$data['note'] ?? null
        ]);
        return redirect()->back()->with('success','Manual time entry added');
    }
}
